% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihood.R
\name{likelihood}
\alias{likelihood}
\title{Estimate the log-likelihood/likelihood for observed branching processes}
\usage{
likelihood(
  chains,
  statistic = c("size", "length"),
  offspring_dist,
  nsim_obs,
  obs_prob = 1,
  stat_threshold = Inf,
  log = TRUE,
  exclude = NULL,
  individual = FALSE,
  ...
)
}
\arguments{
\item{chains}{Vector of chain summaries (sizes/lengths). Can be a
\verb{<numeric>} vector or an object of class \verb{<epichains>} or
\verb{<epichains_summary>} (obtained from \code{\link[=simulate_chains]{simulate_chains()}} or
\code{\link[=simulate_chain_stats]{simulate_chain_stats()}}). See examples below.}

\item{statistic}{The chain statistic to track as the
stopping criteria for each chain being simulated when \code{stat_threshold} is not
\code{Inf}; A \verb{<string>}. It can be one of:
\itemize{
\item "size": the total number of cases produced by a chain before it goes
extinct.
\item "length": the total number of generations reached by a chain before
it goes extinct.
}}

\item{offspring_dist}{Offspring distribution: a \verb{<function>} like the ones
provided by R to generate random numbers from given distributions (e.g.,
\code{\link{rpois}} for Poisson). More specifically, the function needs to
accept at least one argument, \code{n}, which is the number of random
numbers to generate. It can accept further arguments, which will be passed
on to the random number generating functions. Examples that can be provided
here are \code{rpois} for Poisson distributed offspring, \code{rnbinom} for negative
binomial offspring, or custom functions.}

\item{nsim_obs}{Number of simulations to be used to approximate the
log-likelihood/likelihood if \code{obs_prob < 1} (imperfect observation). If
\code{obs_prob == 1}, this argument is ignored.}

\item{obs_prob}{Observation probability. A number (probability) between 0
and 1. A value greater than 0 but less 1 implies imperfect observation and
simulations will be used to approximate the (log)likelihood. This will
also require specifying \code{nsim_obs}. In the simulation, the observation
process is assumed to be constant.}

\item{stat_threshold}{A stopping criterion for individual chain simulations;
a positive number coercible to integer. When any chain's cumulative statistic
reaches or surpasses \code{stat_threshold}, that chain ends. It also serves as a
censoring limit so that results above the specified value, are set to \code{Inf}.
Defaults to \code{Inf}. NOTE: For objects of class \verb{<epichains>} or
\verb{<epichains_summary>}, the \code{stat_threshold} used in the simulation is
extracted and used here so if this argument is specified, it is ignored and
a warning is thrown.}

\item{log}{Should the log-likelihoods be transformed to
likelihoods? Logical. Defaults to \code{TRUE}.}

\item{exclude}{A vector of indices of the sizes/lengths to exclude from the
log-likelihood calculation.}

\item{individual}{Logical; If TRUE, a vector of individual
log-likelihood/likelihood contributions will be returned rather than the
sum/product.}

\item{...}{any parameters to pass to \code{\link{simulate_chain_stats}}}
}
\value{
If \code{log = TRUE}
\itemize{
\item A joint log-likelihood (sum of individual log-likelihoods), if
\code{individual == FALSE} (default) and \code{obs_prob == 1} (default), or
\item A list of individual log-likelihoods, if \code{individual == TRUE} and
\code{obs_prob == 1} (default), or
\item A list of individual log-likelihoods (same length as \code{nsim_obs}), if
\code{individual == TRUE} and \code{0 <= obs_prob < 1}, or
\item A vector of joint log-likelihoods (same length as \code{nsim_obs}), if
individual == FALSE and \code{0 <= obs_prob < 1} (imperfect observation).
}

If \code{log = FALSE}, the same structure of outputs as above are returned,
except that likelihoods, instead of log-likelihoods, are calculated in all
cases. Moreover, the joint likelihoods are the product, instead of the sum,
of the individual likelihoods.
}
\description{
Estimate the log-likelihood/likelihood for observed branching processes
}
\examples{
# example of observed chain sizes
set.seed(121)
# randomly generate 20 chains of size 1 to 10
chain_sizes <- sample(1:10, 20, replace = TRUE)
likelihood(
  chains = chain_sizes,
  statistic = "size",
  offspring_dist = rpois,
  nsim_obs = 100,
  lambda = 0.5
)
# Example using an <epichains> object
set.seed(32)
epichains_obj_eg <- simulate_chains(
  n_chains = 10,
  pop = 100,
  percent_immune = 0,
  statistic = "size",
  offspring_dist = rnbinom,
  stat_threshold = 10,
  generation_time = function(n) rep(3, n),
  mu = 2,
  size = 0.2
)

epichains_obj_eg_lik <- likelihood(
  chains = epichains_obj_eg,
  statistic = "size",
  offspring_dist = rnbinom,
  mu = 2,
  size = 0.2,
  stat_threshold = 10
)
epichains_obj_eg_lik

# Example using a <epichains_summary> object
set.seed(32)
epichains_summary_eg <- simulate_chain_stats(
  n_chains = 10,
  pop = 100,
  percent_immune = 0,
  statistic = "size",
  offspring_dist = rnbinom,
  stat_threshold = 10,
  mu = 2,
  size = 0.2
)
epichains_summary_eg_lik <- likelihood(
  chains = epichains_summary_eg,
  statistic = "size",
  offspring_dist = rnbinom,
  mu = 2,
  size = 0.2
)
epichains_summary_eg_lik
}
\author{
Sebastian Funk, James M. Azam
}
