% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/entropy_pooling.R
\name{entropy_pooling}
\alias{entropy_pooling}
\title{Numerical Entropy Minimization}
\usage{
entropy_pooling(
  p,
  A = NULL,
  b = NULL,
  Aeq = NULL,
  beq = NULL,
  solver = c("nlminb", "solnl", "nloptr"),
  ...
)
}
\arguments{
\item{p}{A vector of prior probabilities.}

\item{A}{The linear inequality constraint (left-hand side).}

\item{b}{The linear inequality constraint (right-hand side).}

\item{Aeq}{The linear equality constraint (left-hand side).}

\item{beq}{The linear equality constraint (right-hand side).}

\item{solver}{A \code{character}. One of: "nlminb", "solnl" or "nloptr".}

\item{...}{Further arguments passed to one of the solvers.}
}
\value{
A vector of posterior probabilities.
}
\description{
This function solves the entropy minimization problem with equality and inequality
constraints. The solution is a vector of posterior probabilities that distorts
the least the prior (equal-weights probabilities) given the constraints (views on
the market).
}
\details{
When imposing views constraints there is no need to specify the non-negativity
constraint for probabilities, which is done automatically by \code{entropy_pooling}.

For the arguments accepted in \code{...}, please see the documentation of
\code{\link[stats]{nlminb}}, \code{\link[NlcOptim]{solnl}}, \code{\link[nloptr]{nloptr}}
and the examples bellow.
}
\examples{
# setup
ret <- diff(log(EuStockMarkets))
n   <- nrow(ret)

# View on expected returns (here is 2\% for each asset)
mean <- rep(0.02, 4)

# Prior probabilities (usually equal weight scheme)
prior <- rep(1 / n, n)

# View
views <- view_on_mean(x = ret, mean = mean)

# Optimization
ep <- entropy_pooling(
 p      = prior,
 Aeq    = views$Aeq,
 beq    = views$beq,
 solver = "nlminb"
)
ep

### Using the ... argument to control the optimization parameters

# nlminb
ep <- entropy_pooling(
 p      = prior,
 Aeq    = views$Aeq,
 beq    = views$beq,
 solver = "nlminb",
 control = list(
     eval.max = 1000,
     iter.max = 1000,
     trace    = TRUE
   )
)
ep

# nloptr
ep <- entropy_pooling(
 p      = prior,
 Aeq    = views$Aeq,
 beq    = views$beq,
 solver = "nloptr",
 control = list(
     xtol_rel = 1e-10,
     maxeval  = 1000,
     check_derivatives = TRUE
   )
)
ep
}
