% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fic.R
\name{fic_core}
\alias{fic_core}
\alias{fic_multi}
\title{Focused information criteria: core calculation functions}
\usage{
fic_core(par, J, inds, inds0, gamma0 = 0, n, focus_deriv = NULL)

fic_multi(
  par,
  J,
  inds,
  inds0,
  gamma0 = 0,
  n,
  focus = NULL,
  focus_deriv = NULL,
  wt = NULL,
  parsub = NULL,
  auxpar = NULL,
  auxsub = NULL,
  ...
)
}
\arguments{
\item{par}{Vector of maximum likelihood estimates from the wide model}

\item{J}{Information matrix from the wide model, evaluated at the maximum likelihood estimates (note that this definition differs from Claeskens and Hjort, where \code{J} is defined as the information divided by the sample size \code{n})}

\item{inds}{Matrix or vector of indicators for which parameters are included in the submodel or submodels to be assessed.

A matrix should be supplied if there are multiple submodels.  This should have number of rows equal to the number of submodels, and number of columns equal to the total number of parameters in the wide model.  It contains 1s in the positions where the parameter is included in the submodel, and 0s in positions where the parameter is excluded.  This should always be 1 in the positions defining the narrow model, as specified in \code{inds0}.}

\item{inds0}{Vector of indicators specifying the narrow model, in the same format as \code{inds}.  If this is omitted, the narrow model is assumed to be defined by the first row of \code{inds} (if \code{inds} is a matrix), or \code{inds} itself if this is a vector.}

\item{gamma0}{Vector of special values taken by the parameters \eqn{gamma} which define the narrow model.

This defaults to 0, as in covariate selection, where "excluded" coefficients are fixed to 0. 

This should either be a scalar, assumed to be the same for all parameters fixed in the narrow model, or a vector of length equal to the number of parameters from the wide model which are fixed in the narrow model, that is, the number of entries of \code{inds0} which are zero.}

\item{n}{Number of observations in the data used to fit the wide model.}

\item{focus_deriv}{Vector of partial derivatives of the focus function with respect to the parameters in the wide model.  This is required by \code{fic_core}.

If there are multiple submodels, this should be a matrix with number of rows equal to the number of submodels, and number of columns equal to the number of parameters in the wide model.  If there is a single submodel, this should be a vector with number of columns equal to the number of parameters in the wide model.

This should take the value given by \code{gamma0} in positions where the parameter is excluded from the submodel.
For example, coefficients of covariates should take the value 0 if the covariate is excluded.}

\item{focus}{An R function with:

\itemize{
\item first argument named \code{par}, denoting a vector of parameters, of the same length as in wide model

\item other arguments defining alternative focuses.  These are supplied through the \code{...} argument to \code{\link{fic}}.  In the built-in examples, there is an argument named \code{X}, denoting alternative covariate values.  The required format is documented below. 
}

The function should return the focus quantity of interest.  If additional arguments are supplied which are vectors or matrices, e.g. \code{X}, then these are assumed to represent multiple focuses, and \code{focus} should return a vector giving the focus for \code{par} and each row of \code{X}.  Otherwise \code{focus} should return a scalar giving the focus value at \code{par}.

Not required if \code{focus_deriv} is specified.

Alternatively, \code{focus} can be a character string naming a built-in focus function supplied by the \pkg{fic} package.  Currently these include:

\code{"prob_logistic"}, the probability of the outcome in a logistic regression model

\code{"mean_normal"} the mean outcome in a normal linear regression model

See \code{\link{focus_fns}} for the functions underlying these built-in focuses.}

\item{wt}{Vector of weights to apply to different covariate values in \code{X}.  This should have length equal to the number of alternative values for the covariates, that is, the number of alternative focuses of interest.  The covariate-specific focused model comparison statistics are then supplemented by averaged statistics for a population defined by this distribution of covariate values.  If this argument is omitted, the values are assumed to have equal weight when computing the average.   The weights are not normalised, though the interpretation is unclear if the weights don't sum to one.}

\item{parsub}{Vector of maximum likelihood estimates from the submodel, or a matrix if there are multiple submodels. 
Only required to return the estimate of the focus quantity alongside 
the model assessment statistics for the submodel. If omitted, the estimate is omitted.}

\item{auxpar}{Estimates of auxiliary parameters from the wide model. The only built-in example is the dispersion parameter for GLMs.}

\item{auxsub}{List of estimates of auxiliary parameters from the submodel.  The only built-in example is the dispersion parameter for GLMs.}

\item{...}{Other arguments to the focus function can be supplied here.

The built-in focus functions \code{\link{prob_logistic}} and \code{\link{mean_normal}} take an argument \code{X} giving covariate values defining the focus. This can either be a matrix or a vector, or a list or data frame that can be coerced into a matrix. 

If just one focus is needed, then \code{X} can be a vector of length equal to the number of parameters in the wide model.

To compute focused model comparison statistics for multiple focuses defined by the same focus function evaluated at multiple covariate values, \code{X} should be a matrix, with number of columns equal to the number of parameters in the wide model, and number of rows equal to the number of alternative focuses.

For a typical regression model, the first parameter will denote an intercept, so the first value of \code{X} should be 1, and the remaining values should correspond to covariates whose coefficients form parameters of the wide model.  See the examples in the vignette.

Arguments to the focus function other than \code{X} can also be supplied as a matrix, vector, list or data frame in the same way. An exception is when the argument is supplied as a vector, this is assumed to refer to multiple focuses.   For example, suppose the focus function defines the quantile of a distribution, and takes an argument \code{focus_p}, then calling \code{fic(...,focus_p=c(0.1, 0.9))} indicates two alternative focuses defined by the 0.1 and 0.9 quantiles.}
}
\value{
See \code{\link{fic}}
}
\description{
Core FIC calculation functions underlying the user interface in \code{\link{fic}}.
\code{fic_core} just handles one submodel, while \code{fic_multi} can assess multiple submodels of the same wide model.  For \code{fic_multi}, \code{inds} and \code{parsub} can be matrices with one row per submodel, while for \code{fic_core} they must be vectors.
}
\examples{

## Lower-level implementation of the example in the main vignette

wide.glm <- glm(low ~ lwtkg + age + smoke + ht + ui + smokeage + smokeui,
                data=birthwt, family=binomial)
mod1.glm <- glm(low ~ lwtkg + age + smoke, data=birthwt, family=binomial)
inds0 <- c(1,1,0,0,0,0,0,0)
inds1 <- c(1,1,1,1,0,0,0,0)
focus_plogis <- function(par, X)plogis(X \%*\% par)
vals.smoke <-    c(1, 58.24, 22.95, 1, 0, 0, 22.95, 0)
vals.nonsmoke <- c(1, 59.50, 23.43, 0, 0, 0, 0, 0)
X <- rbind(vals.smoke, vals.nonsmoke)
par <- coef(wide.glm)
n <- nrow(birthwt)
J <- solve(vcov(wide.glm))
fic_multi(par=par, J=J,  inds=inds1, inds0=inds0, n=n, focus="prob_logistic",
          X=X, parsub=c(coef(mod1.glm), 0, 0, 0, 0))

## Even lower-level implementation, requiring derivatives of the focus
## These are available analytically in this example, but would normally
## need to be calculated using numerical differentiation

focus_deriv <- prob_logistic_deriv(par=par, X=X)
fic_core(par=par, J=J, inds=inds1, inds0=inds0, gamma0=0, n=n,
         focus_deriv=focus_deriv)

}
\seealso{
\code{\link{fic}}
}
