% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fmrReg.R
\name{path.fmrReg}
\alias{path.fmrReg}
\title{Finite Mixture Model with lasso and adaptive penalty}
\usage{
path.fmrReg(y, X, m, equal.var = FALSE,
            ic.type = "ALL", B = NULL, prob = NULL, rho = NULL, 
            control = list(), modstr = list(), report = FALSE)
}
\arguments{
\item{y}{a vector of response (\eqn{n \times 1})}

\item{X}{a matrix of covariate (\eqn{n \times p})}

\item{m}{number of components}

\item{equal.var}{indicating whether variances of different components are equal}

\item{ic.type}{the information criterion to be used; currently supporting "ALL", "AIC", "BIC", and "GIC".}

\item{B}{initial values for the rescaled coefficients with columns being 
the columns being the coefficient for different components}

\item{prob}{initial values for prior probabilitis for different components}

\item{rho}{initial values for rho vector (\eqn{1 / \sigma}), the reciprocal of standard deviation}

\item{control}{a list of parameters for controlling the fitting process}

\item{modstr}{a list of model parameters controlling the model fitting}

\item{report}{indicating whether printing the value of objective function during EM algorithm 
for validation checking of initial value.}
}
\value{
A list consisting of
\item{lambda}{vector of lambda used in model fitting}
\item{B.hat}{estimated rescaled coefficient (\eqn{p \times m \times nlambda})}
\item{pi.hat}{estimated prior probabilities (\eqn{m \times nlambda})}
\item{rho.hat}{estimated rho values (\eqn{m \times nlambda})}
\item{IC}{values of information criteria}
}
\description{
Produce solution paths of regularized finite mixture model with lasso or 
adaptive lasso penalty; compute the degrees of freeom, likelihood 
and information criteria (AIC, BIC and GIC) of the estimators. 
Model fitting is conducted by EM algorithm and coordinate descent.
}
\details{
Model parameters can be specified through argument \code{modstr}. The
available include
\itemize{
   \item{lambda}: A vector of user-specified lambda values with default NULL.
   
   \item{lambda.min.ratio}: Smallest value for lambda, as a fraction of lambda.max, 
       the (data derived) entry value.
   
   \item{nlambda}: The number of lambda values.
   
   \item{w}: Weight matrix for penalty function. Default option is NULL, which means 
       lasso penailty is used for model fitting.
   
   \item{intercept}: Should intercept(s) be fitted (default=TRUE) or set to zero (FALSE).
   
   \item{no.penalty}: A vector of user-specified indicators of the variables
       with no penalty.
   
   \item{common.var}: A vector of user-specified indicators of the variables
       with common effect among different components.
   
   \item{select.ratio}: A user-specified ratio indicating the ratio of variables to be selected.
}

The available elements for argument \code{control} include
\itemize{
   \item{epsilon}: Convergence threshold for generalized EM algorithm.
       Defaults value is 1E-6.
   
   \item{maxit}: Maximum number of passes over the data for all lambda values.
       Default is 1000.
   
   \item{inner.maxit}: Maximum number of iteration for flexmix package to compute initial values.
       Defaults value is 200.
   
   \item{n.ini}: Number of initial values for EM algorithm. Default is 10. In EM algorithm, it is 
       preferable to start from several different initial values.
}
}
\examples{
\donttest{
library(fmerPack)
## problem settings
n <- 100; m <- 3; p <- 5;
sigma2 <- c(0.1, 0.1, 0.4); rho <- 1 / sqrt(sigma2)
phi <- rbind(c(1, 1, 1), c(1, 1, 1), c(1, 1, 1), c(-3, 3, 0), c(3, 0, -3))
beta <- t(t(phi) / rho)
## generate response and covariates
z <- rmultinom(n, 1, prob= rep(1 / m, m))
X <- matrix(rnorm(n * p), nrow = n, ncol = p)
y <- MASS::mvrnorm(1, mu = rowSums(t(z) * X[, 1:(nrow(beta))] \%*\% beta), 
                   Sigma = diag(colSums(z * sigma2)))
## lasso
fit1 <- path.fmrReg(y, X, m = m, modstr = list(nlambda = 10), control = list(n.ini = 1))
## adaptive lasso
fit2 <- path.fmrReg(y, X, m = m, 
                   modstr = list(w = abs(select.tuning(fit1)$B + 1e-6)^2))
}
}
