% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/general_operations.R
\name{spa_contour}
\alias{spa_contour}
\title{Capture the frontier of a plateau region object}
\usage{
spa_contour(pregion)
}
\arguments{
\item{pregion}{A \code{pregion} object. It throws an error if a different type is given.}
}
\value{
A \code{pline} object that represents the contour (i.e. frontier) of a plateau region object given as input.
}
\description{
\code{spa_contour()} extracts the frontier (i.e., linear boundary) of a plateau region object by maintaining its membership degrees.
}
\details{
The \code{spa_contour()} function implements the definition of \emph{fuzzy frontier} of a fuzzy region object in the context of Spatial Plateau Algebra.
The \emph{fuzzy frontier} of a fuzzy region object \code{A} collects all single points of \code{A}, preserving its membership degrees, that are not in the interior of its support.

Note that fuzzy frontier is different from fuzzy boundary (see \code{spa_boundary()}).
}
\examples{
library(tibble)
library(sf)
library(ggplot2)

# defining two different types of membership functions
trap_mf <- function(a, b, c, d) {
  function(x) {
    pmax(pmin((x - a)/(b - a), 1, (d - x)/(d - c), na.rm = TRUE), 0)
  }
}

set.seed(7)
tbl = tibble(x = runif(20, min = 0, max = 30), 
             y = runif(20, min = 0, max = 50), 
             z = runif(20, min = 0, max = 100))
classes <- c("cold", "hot")
cold_mf <- trap_mf(0, 10, 20, 35)
hot_mf <- trap_mf(20, 50, 100, 100)

# Getting the convex hull on the points to clip plateau region objects during their constructions
pts <- st_as_sf(tbl, coords = c(1, 2))
ch <- st_convex_hull(do.call(c, st_geometry(pts)))

# Using the standard fuzzification policy based on fuzzy sets
pregions <- spa_creator(tbl, classes = classes, mfs = c(cold_mf, hot_mf), base_poly = ch)
pregions
\dontrun{
plot(pregions$pgeometry[[1]]) + ggtitle("Cold")
plot(pregions$pgeometry[[2]]) + ggtitle("Hot")
}
# capturing and showing the frontier of each pgeometry object previously created
cold_contour <- spa_contour(pregions$pgeometry[[1]])
hot_contour <- spa_contour(pregions$pgeometry[[2]])
\dontrun{
plot(cold_contour) + ggtitle("Frontier (Cold)")
plot(hot_contour) + ggtitle("Frontier (Hot)")
}
}
\references{
\itemize{
\item \href{https://ieeexplore.ieee.org/document/7737976}{Carniel, A. C.; Schneider, M. A Conceptual Model of Fuzzy Topological Relationships for Fuzzy Regions. In Proceedings of the 2016 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2016), pp. 2271-2278, 2016.}
\item \href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
}
}
