% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggbiplot.r
\name{ggbiplot}
\alias{ggbiplot}
\title{Biplot for Principal Components using ggplot2}
\usage{
ggbiplot(
  pcobj,
  choices = 1:2,
  scale = 1,
  pc.biplot = TRUE,
  obs.scale = 1 - scale,
  var.scale = scale,
  var.factor = 1,
  groups = NULL,
  point.size = 1.5,
  ellipse = FALSE,
  ellipse.prob = 0.68,
  ellipse.linewidth = 1.3,
  ellipse.fill = TRUE,
  ellipse.alpha = 0.25,
  labels = NULL,
  labels.size = 3,
  alpha = 1,
  var.axes = TRUE,
  circle = FALSE,
  circle.prob = 0.68,
  varname.size = 3,
  varname.adjust = 1.25,
  varname.color = "black",
  varname.abbrev = FALSE,
  axis.title = "PC",
  ...
)
}
\arguments{
\item{pcobj}{an object returned by \code{\link[stats]{prcomp}}, \code{\link[stats]{princomp}}, 
\code{\link[FactoMineR]{PCA}}, \code{\link[ade4]{dudi.pca}}, or \code{\link[MASS]{lda}}}

\item{choices}{Which components to plot? An integer vector of length 2.}

\item{scale}{Covariance biplot (\code{scale = 1}), form biplot (\code{scale = 0}). 
When \code{scale = 1} (the default), the inner product 
between the variables approximates the covariance and the distance between the points 
approximates the Mahalanobis distance.}

\item{pc.biplot}{Logical, for compatibility with \code{biplot.princomp()}. If \code{TRUE}, use what Gabriel (1971) 
refers to as a "principal component biplot", with \eqn{\alpha = 1} and observations scaled 
up by \eqn{sqrt(n)} and variables scaled down by \eqn{sqrt(n)}. Then inner products between 
variables approximate covariances and distances between observations approximate 
Mahalanobis distance.}

\item{obs.scale}{Scale factor to apply to observations}

\item{var.scale}{Scale factor to apply to variables}

\item{var.factor}{Factor to be applied to variable vectors after scaling. This allows the variable vectors to be reflected
(\code{var.factor = -1}) or expanded in length (\code{var.factor > 1}) for greater visibility.
\code{\link{reflect}} provides a simpler way to reflect the variables.}

\item{groups}{Optional factor variable indicating the groups that the observations belong to. 
If provided the points will be colored according to groups and this allows data ellipses also
to be drawn when \code{ellipse = TRUE}.}

\item{point.size}{Size of observation points.}

\item{ellipse}{Logical; draw a normal data ellipse for each group?}

\item{ellipse.prob}{Coverage size of the data ellipse in Normal probability}

\item{ellipse.linewidth}{Thickness of the line outlining the ellipses}

\item{ellipse.fill}{Logical; should the ellipses be filled?}

\item{ellipse.alpha}{Transparency value (0 - 1) for filled ellipses}

\item{labels}{Optional vector of labels for the observations. Often, this will be specified as the \code{row.names()}
of the dataset.}

\item{labels.size}{Size of the text used for the point labels}

\item{alpha}{Alpha transparency value for the points (0 = transparent, 1 = opaque)}

\item{var.axes}{logical; draw arrows for the variables?}

\item{circle}{draw a correlation circle? (only applies when prcomp was called with 
\code{scale = TRUE} and when \code{var.scale = 1})}

\item{circle.prob}{Size of the correlation circle}

\item{varname.size}{Size of the text for variable names}

\item{varname.adjust}{Adjustment factor the placement of the variable names, >= 1 means farther from the arrow}

\item{varname.color}{Color for the variable vectors and names}

\item{varname.abbrev}{logical; whether or not to abbreviate the variable names, using \code{\link{abbreviate}}.}

\item{axis.title}{character; the prefix used as the axis labels. Default: \code{"PC"}.}

\item{...}{other arguments passed down}
}
\value{
a ggplot2 plot object of class \code{c("gg", "ggplot")}
}
\description{
A biplot simultaneously displays information on the observations (as points)
and the variables (as vectors) in a multidimensional dataset. The 2D biplot
is typically based on the first two principal components of a dataset, giving a rank 2 approximation 
to the data. The “bi” in biplot refers to the fact that two sets of points (i.e., the rows and
columns of the data matrix) are visualized by scalar products, not the fact
that the display is usually two-dimensional.

The biplot method for principal component analysis was originally defined by Gabriel (1971, 1981).
Gower & Hand (1996) give a more complete treatment. Greenacre (2010) is a practical user-oriented guide to biplots.
Gower et al. (2011) is the most up to date
exposition of biplot methodology.

This implementation handles the results of a principal components analysis using 
\code{\link[stats]{prcomp}}, \code{\link[stats]{princomp}}, \code{\link[FactoMineR]{PCA}} and \code{\link[ade4]{dudi.pca}};
also handles a discriminant analysis using \code{\link[MASS]{lda}}.
}
\details{
The biplot is constructed by using the singular value decomposition (SVD) to obtain a low-rank 
approximation to the data matrix \eqn{\mathbf{X}_{n \times p}} (centered, and optionally scaled to unit variances)
whose \eqn{n} rows are the observations 
and whose \eqn{p} columns are the variables. 

Using the SVD, the matrix \eqn{\mathbf{X}}, of rank \eqn{r \le p}
can be expressed \emph{exactly} as
\deqn{\mathbf{X} = \mathbf{U} \mathbf{\Lambda} \mathbf{V}'
                 = \Sigma_i^r \lambda_i \mathbf{u}_i \mathbf{v}_i' \; ,}

where 
\itemize{
   \item \eqn{\mathbf{U}} is an \eqn{n \times r} orthonormal matrix of observation scores; these are also the eigenvectors
         of \eqn{\mathbf{X} \mathbf{X}'},
   \item \eqn{\mathbf{\Lambda}} is an \eqn{r \times r} diagonal matrix of singular values, 
         \eqn{\lambda_1 \ge \lambda_2 \ge \cdots \lambda_r} 
        % which are also the square roots
        % of the eigenvalues of \eqn{\mathbf{X} \mathbf{X}'}. 
   \item \eqn{\mathbf{V}} is an \eqn{r \times p} orthonormal matrix of variable weights and also the eigenvectors
         of \eqn{\mathbf{X}' \mathbf{X}}.
}

Then, a rank 2 (or 3) PCA approximation \eqn{\widehat{\mathbf{X}}} to the data matrix used in the biplot
can be obtained from the first 2 (or 3)
singular values \eqn{\lambda_i}
and the corresponding \eqn{\mathbf{u}_i, \mathbf{v}_i} as

\deqn{\mathbf{X} \approx \widehat{\mathbf{X}} = \lambda_1 \mathbf{u}_1 \mathbf{v}_1' + \lambda_2 \mathbf{u}_2 \mathbf{v}_2' \; .}

The variance of \eqn{\mathbf{X}} accounted for by each term is \eqn{\lambda_i^2}.

The biplot is then obtained by overlaying two scatterplots that share a common set of axes and have a between-set scalar 
product interpretation. Typically, the observations (rows of \eqn{\mathbf{X}}) are represented as points
and the variables (columns of \eqn{\mathbf{X}}) are represented as vectors from the origin.

The \code{scale} factor, \eqn{\alpha} allows the variances of the components to be apportioned between the
row points and column vectors, with different interpretations, by representing the approximation
\eqn{\widehat{\mathbf{X}}} as the product of two matrices,

\deqn{\widehat{\mathbf{X}} = (\mathbf{U} \mathbf{\Lambda}^\alpha) (\mathbf{\Lambda}^{1-\alpha} \mathbf{V}')}

The choice \eqn{\alpha = 1}, assigning the singular values totally to the left factor,
 gives a distance interpretation to the row display and 
\eqn{\alpha = 0} gives a distance interpretation to the column display.
\eqn{\alpha = 1/2} gives a symmetrically scaled biplot.

When the singular values are assigned totally to the left or to the right factor, the resultant 
coordinates are called \emph{principal coordinates} and the sum of squared coordinates
on each dimension equal the corresponding singular value.
The other matrix, to which no part of the singular 
values is assigned, contains the so-called \emph{standard coordinates} and have sum of squared
values equal to 1.0.
}
\examples{
data(wine)
library(ggplot2)
wine.pca <- prcomp(wine, scale. = TRUE)
ggbiplot(wine.pca, 
         obs.scale = 1, var.scale = 1, 
         varname.size = 4,
         groups = wine.class, 
         ellipse = TRUE, circle = TRUE)

data(iris)
iris.pca <- prcomp (~ Sepal.Length + Sepal.Width + Petal.Length + Petal.Width,
                    data=iris,
                    scale. = TRUE)
ggbiplot(iris.pca, obs.scale = 1, var.scale = 1,
         groups = iris$Species, point.size=2,
         varname.size = 5, 
         varname.color = "black",
         varname.adjust = 1.2,
         ellipse = TRUE, 
         circle = TRUE) +
  labs(fill = "Species", color = "Species") +
  theme_minimal(base_size = 14) +
  theme(legend.direction = 'horizontal', legend.position = 'top')
}
\references{
Gabriel, K. R. (1971). The biplot graphical display of matrices with application to principal component analysis. 
  \emph{Biometrika}, \bold{58}, 453–467. \doi{10.2307/2334381}.
  
  Gabriel, K. R. (1981). Biplot display of multivariate matrices for inspection of data and diagnosis. 
  In V. Barnett (Ed.), \emph{Interpreting Multivariate Data}. London: Wiley. 
  
  Greenacre, M. (2010). \emph{Biplots in Practice}. BBVA Foundation, Bilbao, Spain. 
  Available for free at \url{https://www.fbbva.es/microsite/multivariate-statistics/}.
  
  J.C. Gower and D. J. Hand (1996). \emph{Biplots}. Chapman & Hall.
  
  Gower, J. C., Lubbe, S. G., & Roux, N. J. L. (2011). \emph{Understanding Biplots}. Wiley.
}
\seealso{
\code{\link{reflect}}, \code{\link{ggscreeplot}};
  \code{\link[stats]{biplot}} for the original stats package version;
  \code{\link[factoextra]{fviz_pca_biplot}} for the factoextra package version.
}
\author{
Vincent Q. Vu.
}
