% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/means.R
\name{extended_mean}
\alias{extended_mean}
\alias{generalized_logmean}
\alias{logmean}
\title{Extended mean}
\usage{
extended_mean(r, s)

generalized_logmean(r)

logmean(a, b, tol = .Machine$double.eps^0.5)
}
\arguments{
\item{r, s}{A finite number giving the order of the generalized logarithmic
mean / extended mean.}

\item{a, b}{A strictly positive numeric vector.}

\item{tol}{The tolerance used to determine if \code{a == b}.}
}
\value{
\code{generalized_logmean()} and \code{extended_mean()} return a
function:

\preformatted{function(a, b, tol = .Machine$double.eps^0.5){...}}

\code{logmean()} returns a numeric vector, the same length as
\code{max(length(a), length(b))}, giving the component-wise logarithmic mean
of \code{a} and \code{b}.
}
\description{
Calculate a generalized logarithmic mean / extended mean.
}
\details{
The function \code{extended_mean()} returns a function to compute the
component-wise extended mean of \code{a} and \code{b} of orders \code{r} and
\code{s}. See Bullen (2003, p. 393) for a definition. This is also called
the difference mean, Stolarsky mean, or extended mean-value mean.

The function \code{generalized_logmean()} returns a function to compute the
component-wise generalized logarithmic mean of \code{a} and \code{b} of
order \code{r}. See Bullen (2003, p. 385) for a definition. The generalized
logarithmic mean is a special case of the extended mean, corresponding to
\code{extended_mean(r, 1)()}, but is more commonly used for price indexes.

The function \code{logmean()} returns the ordinary component-wise
logarithmic mean of \code{a} and \code{b}, and corresponds to
\code{generalized_logmean(1)()}.

Both \code{a} and \code{b} should be strictly positive. This is not
enforced, but the results may not make sense when the generalized
logarithmic mean / extended mean is not defined. The usual recycling rules
apply when \code{a} and \code{b} are not the same length.

By definition, the generalized logarithmic mean / extended mean of \code{a}
and \code{b} is \code{a} when \code{a == b}. The \code{tol} argument is used
to test equality by checking if \code{abs(a - b) <= tol}. The default value
is the same as \code{\link[=all.equal]{all.equal()}}. In some cases it's useful to multiply
\code{tol} by a scale factor, such as \code{max(abs(a), abs(b))}. This often
doesn't matter when making price indexes, however, as \code{a} and \code{b}
are usually around 1.
}
\note{
\code{generalized_logmean()} can be defined on the extended real line,
so that \code{r = -Inf / Inf} returns \code{\link[=pmin]{pmin()}}/\code{\link[=pmax]{pmax()}}, to agree with the
definition in, e.g., Bullen (2003). This is not implemented, and \code{r}
must be finite.
}
\examples{
x <- 8:5
y <- 1:4

# The arithmetic and geometric means are special cases of the
# generalized logarithmic mean.

all.equal(generalized_logmean(2)(x, y), (x + y) / 2)
all.equal(generalized_logmean(-1)(x, y), sqrt(x * y))

# The harmonic mean cannot be expressed as a logarithmic mean, but can
# be expressed as an extended mean.

all.equal(extended_mean(-2, -1)(x, y), 2 / (1 / x + 1 / y))

# The quadratic mean is also a type of extended mean.

all.equal(extended_mean(2, 4)(x, y), sqrt(x^2 / 2 + y^2 / 2))

# As are heronian and centroidal means.

all.equal(
  extended_mean(0.5, 1.5)(x, y),
  (x + sqrt(x * y) + y) / 3
)
all.equal(
  extended_mean(2, 3)(x, y),
  2 / 3 * (x^2 + x * y + y^2) / (x + y)
)

}
\references{
Bullen, P. S. (2003). \emph{Handbook of Means and Their Inequalities}.
Springer Science+Business Media.
}
\seealso{
\code{\link[=transmute_weights]{transmute_weights()}} uses the extended mean to turn a generalized
mean of order \eqn{r} into a generalized mean of order \eqn{s}.

Other means: 
\code{\link{generalized_mean}()},
\code{\link{lehmer_mean}()},
\code{\link{nested_mean}()}
}
\concept{means}
