% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MVSKtilting.R
\name{design_MVSKtilting_portfolio_via_sample_moments}
\alias{design_MVSKtilting_portfolio_via_sample_moments}
\title{Design high-order portfolio by tilting a given portfolio to the MVSK efficient frontier}
\usage{
design_MVSKtilting_portfolio_via_sample_moments(
  d = rep(1, 4),
  X_moments,
  w_init = rep(1/length(X_moments$mu), length(X_moments$mu)),
  w0 = w_init,
  w0_moments = NULL,
  leverage = 1,
  kappa = 0,
  method = c("Q-MVSKT", "L-MVSKT"),
  tau_w = 1e-05,
  tau_delta = 1e-05,
  gamma = 1,
  zeta = 1e-08,
  maxiter = 100,
  ftol = 1e-05,
  wtol = 1e-05,
  theta = 0.5,
  stopval = -Inf
)
}
\arguments{
\item{d}{Numerical vector of length 4 indicating the weights of first four moments.}

\item{X_moments}{List of moment parameters, see \code{\link{estimate_sample_moments}()}.}

\item{w_init}{Numerical vector indicating the initial value of portfolio weights.}

\item{w0}{Numerical vector indicating the reference portfolio vector.}

\item{w0_moments}{Numerical vector indicating the reference moments.}

\item{leverage}{Number (>= 1) indicating the leverage of portfolio.}

\item{kappa}{Number indicating the maximum tracking error volatility.}

\item{method}{String indicating the algorithm method, must be one of: "Q-MVSK", "MM", "DC".}

\item{tau_w}{Number (>= 0) guaranteeing the strong convexity of approximating function.}

\item{tau_delta}{Number (>= 0) guaranteeing the strong convexity of approximating function.}

\item{gamma}{Number (0 < gamma <= 1) indicating the initial value of gamma.}

\item{zeta}{Number (0 < zeta < 1) indicating the diminishing paramater of gamma.}

\item{maxiter}{Positive integer setting the maximum iteration.}

\item{ftol}{Positive number setting the convergence criterion of function objective.}

\item{wtol}{Positive number setting the convergence criterion of portfolio weights.}

\item{theta}{Number (0 < theta < 1) setting the combination coefficient when enlarge feasible set.}

\item{stopval}{Number setting the stop value of objective.}
}
\value{
A list containing the following elements:
\item{\code{w}}{Optimal portfolio vector.}
\item{\code{delta}}{Maximum tilting distance of the optimal portfolio.}
\item{\code{cpu_time_vs_iterations}}{Time usage over iterations.}
\item{\code{objfun_vs_iterations}}{Objective function over iterations.}
\item{\code{iterations}}{Iterations index.}
\item{\code{moments}}{Moments of portfolio return at optimal portfolio weights.}
\item{\code{improvement}}{The relative improvement of moments of designed portfolio w.r.t. the reference portfolio.}
}
\description{
Design high-order portfolio by tilting a given portfolio to the MVSK efficient frontier
(i.e., mean, variance, skewness, and kurtosis):
\preformatted{
  minimize     - delta
               m1(w) >= m1(w0) + delta*d1
               m2(w) <= m2(w0) - delta*d2
               m3(w) >= m3(w0) + delta*d3
               m4(w) <= m4(w0) - delta*d4
               (w-w0)'Sigma(w-w0) <= kappa^2
  subject to   ||w||_1 <= leverage, sum(w) == 1.
}
}
\examples{

library(highOrderPortfolios)
data(X50)

# estimate moments
X_moments <- estimate_sample_moments(X50[, 1:10])

# decide problem setting
w0 <- rep(1/10, 10)
w0_moments <- eval_portfolio_moments(w0, X_moments)
d <- abs(w0_moments) 
kappa <- 0.3 * sqrt(w0 \%*\% X_moments$Sgm \%*\% w0)

# portfolio optimization
sol <- design_MVSKtilting_portfolio_via_sample_moments(d, X_moments, w_init = w0, w0 = w0, 
                                                       w0_moments = w0_moments, kappa = kappa)


}
\references{
R. Zhou and D. P. Palomar, "Solving High-Order Portfolios via Successive Convex Approximation Algorithms," 
in \emph{IEEE Transactions on Signal Processing}, vol. 69, pp. 892-904, 2021.
<doi:10.1109/TSP.2021.3051369>.
}
\author{
Rui Zhou and Daniel P. Palomar
}
