% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oos_vintage_forc_general.R
\name{oos_vintage_forc_general}
\alias{oos_vintage_forc_general}
\title{Out-of-sample general model forecast conditioned on vintage forecasts}
\usage{
oos_vintage_forc_general(
  model_function,
  prediction_function,
  data,
  realized,
  time_vec,
  ...,
  estimation_window = NULL
)
}
\arguments{
\item{model_function}{Function that estimates a model using the \code{data} input.}

\item{prediction_function}{Function that generates model predictions using \code{model_function}
and \code{data} arguments. Note* that the \code{data} argument passed to the prediction_function
takes the form of a data.frame with a number of columns equal to the number of
input vintage forecasts passed by the user. The prediction_function needs to be able to take
this input format and generate a prediction based on it.}

\item{data}{Input data for estimating the model.}

\item{realized}{Vector of realized values of the dependent variable equal in length
to the data in \code{data}.}

\item{time_vec}{Vector of any class that represents time and is equal in length to
the length of \code{realized} and \code{data}.}

\item{...}{Set of forecasts of class Forecast, one forecast for each
parameter in the linear model.}

\item{estimation_window}{Integer representing the number of past periods
that the linear model should be estimated over in each period.}
}
\value{
\code{\link{Forecast}} object that contains the out-of-sample
forecast.
}
\description{
\code{oos_vintage_forc_general} takes a model function, a prediction function,
input data for estimating the model, realized values of the dependent variable,
a vector of time data associated with the model, a forecast for each parameter
in the model, and an optional integer number of past periods to estimate the
model over. For each period in the vintage forecasts, model parametes are
estimated with data up to the current period minus the number of periods
specified in \code{estimation_window}. If \code{estimation_window} is left
\code{NULL} then the model is estimated with all available data up to the
current period. Model parameters are then combined with vintage forecast values
to generate a forecast. Returns an out-of-sample forecast conditional on vintage
forecasts that \strong{would} have been available at the forecast origin. Replicates
the forecasts that a conditional forecasting model would have produced in real time.
}
\examples{
# Estimation Data.
date <- as.Date(c("2010-03-31", "2010-06-30", "2010-09-30", "2010-12-31",
                  "2011-03-31", "2011-06-30", "2011-09-30", "2011-12-31", 
                  "2012-03-31", "2012-06-30", "2012-09-30", "2012-12-31",
                  "2013-03-31", "2013-06-30", "2013-09-30", "2013-12-31"))
y  <- c(1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0)
x1 <- c(8.22, 3.86, 4.27, 3.37, 5.88, 3.34, 2.92, 1.80, 3.30, 7.17, 3.22, 3.86, 
    4.27, 3.37, 5.88, 3.34)
x2 <- c(4.03, 2.46, 2.04, 2.44, 6.09, 2.91, 1.68, 2.91, 3.87, 1.63, 4.03, 2.46, 
    2.04, 2.44, 6.09, 2.91)
dataLogit <- data.frame(date, y, x1, x2)

# Vintage Forecasts.
x1_forecast_vintageLogit <- Forecast(
   origin   = as.Date(c("2012-09-30", "2012-12-31", "2013-03-31", "2013-06-30")),
   future   = as.Date(c("2013-09-30", "2013-12-31", "2014-03-31", "2014-06-30")),
   forecast = c(6.34, 4.17, 2.98, 1.84),
   realized = c(5.88, 3.34, 2.92, 1.80),
   h_ahead  = 4L
)

x2_forecast_vintageLogit <- Forecast(
   origin   = as.Date(c("2012-09-30", "2012-12-31", "2013-03-31", "2013-06-30")),
   future   = as.Date(c("2013-09-30", "2013-12-31", "2014-03-31", "2014-06-30")),
   forecast = c(7.32, 3.22, 2.21, 2.65),
   realized = c(6.09, 2.91, 1.68, 2.91),
   h_ahead  = 4L
)

# Forecasting function.
oos_vintage_forc_general(
    model_function = function(data) {glm(y ~ x1 + x2, data = data, family = binomial)},
    prediction_function = function(model_function, data) {
        names(data) <- c("x1", "x2")
        as.vector(predict(model_function, data, type = "response"))
    }, 
    data = dataLogit,
    realized = dataLogit$y,
    time_vec = dataLogit$date,
    x1_forecast_vintageLogit, x2_forecast_vintageLogit,
    estimation_window = NULL
)

}
\seealso{
For a detailed example see the help vignette:
\code{vignette("lmForc", package = "lmForc")}
}
