#' @name PlotData
#'
#' @title Plotting method for \code{survDataVar} objects
#'
#' @description
#' This is the generic \code{plot} S3 method for the \code{survDataVar} class.
#' It plots the number of survivors as a function of time.
#'
#' @param x an object of class \code{survData}.
#' @param xlab a label for the \eqn{X}-axis, by default \code{Time}.
#' @param ylab a label for the \eqn{Y}-axis, by default \code{Number of survivors}.
#' @param main main title for the plot.
#' @param one_plot if \code{TRUE}, draws all the points in one plot instead of
#'  one per \code{replicate} or \code{conc}.
#' @param add_legend if \code{TRUE}, add the legend to the plot.
#' @param \dots Further arguments to be passed to generic methods.
#' 
#' @return an object of class \code{ggplot}, 
#' see function \code{\link[ggplot2]{ggplot}}
#'
#' @keywords plot
#' 
#' @import ggplot2
#'
#' @export
#' 
plot.SurvDataVarExp <- function(x,
                        xlab = "Time",
                        ylab = "Number of survivors",
                        main = NULL,
                        one_plot = FALSE,
                        add_legend = FALSE,
                        ...){
    
    data_plt = x[!is.na(x$Nsurv), ]
    
    plt = ggplot(data_plt,
                 aes(x = time, y = Nsurv,
                     group = replicate,
                     color = conc)) +
        theme_minimal() +
        labs(title = main,
             x = xlab,
             y = ylab,
             colour = "Concentration" # legend title
        ) +
        scale_colour_gradient(
            name = "Concentration",
            low = cexpmin, high = cexpmax ) +
        scale_fill_gradient(
            name = "Concentration",
            low = cexpmin, high = cexpmax ) +
        expand_limits(x = 0, y = 0) +
        geom_point() +
        geom_line()
    
    if (!one_plot) {
        plt <- plt + facet_wrap(~ replicate, ncol = 2)
    }
    if (!add_legend) {
        plt <- plt + theme(legend.position = "none")
    }
    
    return(plt)
}

#' @name PlotData
#' @export
plot.SurvDataCstExp <- function(x,
                          xlab = "Time",
                          ylab = "Number of survivors",
                          main = NULL,
                          one_plot = FALSE,
                          ...){
    
    data_plt = x[!is.na(x$Nsurv), ]
    
    plt = ggplot(data_plt,
                 aes(x = time, y = Nsurv,
                     group = as.factor(conc))
                 ) +
        theme_minimal() +
        labs(title = main,
             x = xlab,
             y = ylab,
        ) +
        expand_limits(x = 0, y = 0) +
        geom_point() +
        geom_line()
    
    if (!one_plot) {
        plt <- plt + facet_wrap(~ conc, ncol = 2)
    }
    
    return(plt)
}


