% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gt_nfl.R
\name{gt_pct_bar}
\alias{gt_pct_bar}
\title{Format Columns of 'gt' Tables as Percentage Bars}
\usage{
gt_pct_bar(
  gt_tbl,
  col_value,
  col_pct,
  ...,
  rows = gt::everything(),
  hide_col_pct = FALSE,
  value_position = c("inline", "above"),
  value_scale = 1L,
  value_padding_left = "0px",
  value_padding_right = "0px",
  value_colors = c("black", "white"),
  value_style.props = list(),
  fill_palette = "hulk",
  fill_palette.reverse = FALSE,
  fill_na.color = "#808080",
  fill_pct.domain = 0:100,
  fill_border.color = "transparent",
  fill_border.radius = "10px",
  fill_height = "100\%",
  fill_style.props = list(),
  background_border.color = "thin solid black",
  background_border.radius = "12px",
  background_fill.color = "#b1b1b1",
  background_fill.width = "100\%",
  background_fill.height = "100\%",
  background_style.props = list()
)
}
\arguments{
\item{gt_tbl}{A table object that is created using the \code{\link[gt:gt]{gt::gt()}} function.}

\item{col_value}{Column name of the value to be printed.}

\item{col_pct}{Column name of percentage values controlling the fill width.
If this is not in a 0 - 100 range, use \code{value_scale} to scale it up.}

\item{...}{These dots are for future extensions and must be empty.}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
form a constraint for extraction. The default \code{\link[gt:everything]{everything()}} results in all
rows in \code{columns} being formatted. Alternatively, we can supply a vector of
row IDs within \code{c()}, a vector of row indices, or a select helper function
(e.g. \code{\link[gt:starts_with]{starts_with()}}, \code{\link[gt:ends_with]{ends_with()}}, \code{\link[gt:contains]{contains()}}, \code{\link[gt:matches]{matches()}},
\code{\link[gt:num_range]{num_range()}}, and \code{\link[gt:everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{hide_col_pct}{If \code{TRUE}, the column in \code{col_pct} will be hidden in the
resulting table.}

\item{value_position}{One of the following:
\itemize{
\item \code{"inline"} : prints the value inside of the bars
\item \code{"above"} : prints the value above the bars
}}

\item{value_scale}{A scaling factor: values from column \code{col_pct} will be
multiplied by \code{value_scale} before proceeding. This is useful if the
underlying data is in a 0 - 1 range, instead of the required 0 - 100 range.}

\item{value_padding_left}{Left padding of the printed text.}

\item{value_padding_right}{Right padding of the printed text.}

\item{value_colors}{One or more colors of the printed text. If this is a
vector of colors and \code{value_position = "inline"}, the function will calculate
color contrast ratios with \link[colorspace:contrast_ratio]{colorspace::contrast_ratio} and, based on this,
decide which of the colors to chose to maximize readability. You can
overwrite the resulting colos with \code{value_style.props}.
NOTE: this uses colors from \code{fill_palette} for contrast ratio calculations
and not from \code{background_fill.color} because it is not trivial to figure
out the actual background of the text (it could overlap with both).}

\item{value_style.props}{A named list of the form \code{list(property = value)}
for enhanced control of the html style property. This can overwrite the
default properties set with the above \code{value_} arguments.}

\item{fill_palette}{The colors that values will be mapped to. This can also
be one of \code{"hulk"}, \code{"hulk_teal"}, or \code{"blue_orange"} which will trigger
internal color palettes. Argument passed on to \link[scales:col_numeric]{scales::col_numeric}.}

\item{fill_palette.reverse}{Whether the vector of colors in \code{fill_palette}
should be reversed. Argument passed on to \link[scales:col_numeric]{scales::col_numeric}.}

\item{fill_na.color}{Fill color in case of \code{NA} values. Argument passed on to
\link[scales:col_numeric]{scales::col_numeric}.}

\item{fill_pct.domain}{The possible values that colors in \code{fill_palette} can
be mapped to.}

\item{fill_border.color}{Border color of color filled area.}

\item{fill_border.radius}{Border radius of color filled area.}

\item{fill_height}{The height of the colored fill bar. Should correspond with
\code{background_fill.height}. This defaults to 100\% which will make sure the bar
height matches text size of the printed value (when
\code{value_position = "inline"}). Please note that \code{value_position = "inline"}
requires an absolute value of \code{fill_height}, (e.g. \verb{5px}), otherwise it will
render as line.}

\item{fill_style.props}{A named list of the form \code{list(property = value)}
for enhanced control of the html style property. This can overwrite the
default properties set with the above \code{fill_} arguments.}

\item{background_border.color}{Border color of background.}

\item{background_border.radius}{Border radius of background.}

\item{background_fill.color}{Fill color of background.}

\item{background_fill.width}{Width of background.}

\item{background_fill.height}{The height of the colored background bar.
Should correspond with \code{fill_height}. This defaults to 100\% which will make
sure the bar height matches text size of the printed value (when
\code{value_position = "inline"}). Please note that \code{value_position = "inline"}
requires an absolute value of \code{background_fill.height}, (e.g. \verb{5px}),
otherwise it will render as line.}

\item{background_style.props}{A named list of the form \code{list(property = value)}
for enhanced control of the html style property. This can overwrite the
default properties set with the above \code{background_} arguments.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Add context to your data by adding a percentile bar to the actual values.
The percentile bar is colored with a color scale based on a user supplied
color palette and the relative width of the bars will be rendered as tooltip.
}
\details{
The function allows extensive styling of the bars and text, either by using
some of the default arguments or, if you want full control, by using the
\verb{*_style.props} lists which give you full control over all style properties.
All styling parameters are interpreted as style properties of a html span tag.
For more information on CSS properties, see
\url{https://www.w3schools.com/cssref/index.php}.
\subsection{Some notes about styling}{

Since this is meant to be an extension of an already existing 'gt' table,
you'll have to do some styling outside of this function, esp. the horizontal
alignment and direction will be controlled by \code{gt::cols_align} (see example).

Make sure to play around with \code{fill_border.radius} and
\code{background_border.radius}. Results will depend on final column width and
percentiles. Very short percentile bars, i.e. small values in \code{col_pct},
might result in bars crossing the border when combined with a
big border radius.

Text alignment depending on the colored bar isn't as easy as one might think.
Try percent values in \code{value_padding_left} or \code{value_padding_right} to avoid
overlapping of text values and the outline of the colored bars.

For more information and examples, see the article that describes how
nflplotR works with the 'gt' package
\url{https://nflplotr.nflverse.com/articles/gt.html}.
}
}
\section{Output of below example}{

\if{html}{\figure{pct_tbl.png}{options: style="width: 66\%;"}}
}

\examples{
library(data.table)

# Make a data.table of mtcars and select only disp and hp
data <- data.table::as.data.table(mtcars)[, list(disp, hp)]

# Add the percentile of hp in the distribution of hp values
data[, pct := round(stats::ecdf(hp)(hp) * 100, 1)]

# set seed to keep it reproducible
set.seed(20)

# take random sample (to avoid a big table) and add the percent bars for hp
# using the percentiles in the pct variable
table <- data[sample(.N, 10)] |>
  gt::gt() |>
  nflplotR::gt_pct_bar(
    "hp", "pct",
    hide_col_pct = FALSE,
    value_padding_left = "10px",
  ) |>
  gt::cols_align("left", hp) |>
  gt::cols_width(hp ~ gt::px(250))
}
\seealso{
The article that describes how nflplotR works with the 'gt' package
\url{https://nflplotr.nflverse.com/articles/gt.html}
}
