% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctd.R
\name{ctdDecimate}
\alias{ctdDecimate}
\title{Decimate a ctd Profile}
\usage{
ctdDecimate(
  x,
  p = 1,
  method = "boxcar",
  rule = 1,
  e = 1.5,
  na.rm = FALSE,
  debug = getOption("oceDebug")
)
}
\arguments{
\item{x}{a \linkS4class{ctd} object.}

\item{p}{pressure increment, or vector of pressures.  In the first case,
pressures from 0dbar to the rounded maximum pressure are used, incrementing by
\code{p} dbars.  If a vector of pressures is given, interpolation is done to
these pressures.}

\item{method}{the method to be used for calculating decimated values.  This may
be a string specifying the method, or a function. In the string case, the
possibilities are as follows.
\itemize{
\item \code{"boxcar"} (based on a local average)
\item \code{"approx"} (based on linear
interpolation between neighboring points, using \code{\link[=approx]{approx()}}
with the \code{rule} argument specified here)
\item \code{"approxML"} as \code{"approx"},
except that a mixed layer is assumed to apply above the top data value; this
is done by setting the \code{yleft} argument to \code{\link[=approx]{approx()}}, and
by calling that function with \verb{rule=c(2, 1))}
\item \code{"lm"} (based on local
regression, with \code{e} setting the size of the local region);
\item \code{"rr"} for the Reiniger and Ross method, carried out with \code{\link[=oce.approx]{oce.approx()}};
\item \code{"unesco"} (for the UNESCO method, carried out with \code{\link[=oce.approx]{oce.approx()}}.
}

On the other hand, if \code{method} is a function, then it must take
two arguments, named \code{data} and \code{parameters}. The first is set to \code{x@data} by
\code{ctdTrim()}. The second is passed directly to the user's function (see
Example 2). The return value from the function must be a logical vector of
the same length as the \code{pressure} data, with TRUE values meaning to keep the
corresponding entries of the \code{data} slot.}

\item{rule}{an integer that is passed to \code{\link[=approx]{approx()}}, in the
case where \code{method} is \code{"approx"}. Note that the default value
for \code{rule} is 1, which will inhibit extrapolation beyond the observed
pressure range. This is a change from the behaviour previous to May 8, 2017,
when a \code{rule} of 2 was used (without stating so as an argument).}

\item{e}{is an expansion coefficient used to calculate the local neighbourhoods
for the \code{"boxcar"} and \code{"lm"} methods.  If \code{e=1}, then the
neighbourhood for the i-th pressure extends from the (\code{i-1})-th pressure to
the (\code{i+1})-th pressure.  At the endpoints it is assumed that the outside
bin is of the same pressure range as the first inside bin.  For other values of
\code{e}, the neighbourhood is expanded linearly in each direction.  If the
\code{"lm"} method produces warnings about "prediction from a rank-deficient
fit", a larger value of \code{"e"} should be used.}

\item{na.rm}{logical value indicating whether to remove NA values
before decimating.  This value is ignored unless \code{method} is
\code{boxcar} in which case it is passed to \code{\link[=binMean1D]{binMean1D()}} which does the
averaging. This parameter was added in February 2024, and the
behaviour of \code{\link[=ctdDecimate]{ctdDecimate()}} prior that date was equivalent
to \code{na.rm=FALSE}, so that is the default value, even though
it is expected that many uses will find using TRUE is more
convenient. See \verb{https://github.com/dankelley/oce/issues/2192}
for more discussion.}

\item{debug}{an integer specifying whether debugging information is
to be printed during the processing. This is a general parameter that
is used by many \code{oce} functions. Generally, setting \code{debug=0}
turns off the printing, while higher values suggest that more information
be printed. If one function calls another, it usually reduces the value of
\code{debug} first, so that a user can often obtain deeper debugging
by specifying higher \code{debug} values.}
}
\value{
A \linkS4class{ctd} object, with pressures that are as set by
the \code{"p"} parameter and all other properties modified appropriately.
}
\description{
Interpolate a CTD profile to specified pressure values.  This is used
by \code{\link[=sectionGrid]{sectionGrid()}}, but is also useful for dealing with individual
CTD/bottle profiles.
}
\details{
The \code{"approx"} and \code{"approxML"} methods may be best for bottle data,
in which the usual task is
to interpolate from a coarse sampling grid to a finer one. The distinction
is that \code{"approxML"} assumes a mixed-layer above the top sample value. For CTD data, the
\code{"boxcar"} method may be the preferred choice, because the task is normally
to sub-sample, and some degree of smoothing is usually desired.  (The
\code{"lm"} method can be quite slow, and its results may be quite similar to those of the
boxcar method.)

For widely-spaced data, a sort of numerical cabeling effect can result when
density is computed based on interpolated salinity and temperature.
See reference 2 for a discussion of this issue and possible solutions.
}
\section{A note about flags}{


Data-quality flags contained within the original object are ignored by this
function, and the returned value contains no such flags.  This is because such
flags represent an assessment of the original data, not of quantities derived
from those data.  This function produces a warning to this effect. The
recommended practice is to use \code{\link[=handleFlags]{handleFlags()}} or some other means to
deal with flags before calling the present function.
}

\examples{
library(oce)
data(ctd)
plotProfile(ctd, "salinity", ylim = c(10, 0))
p <- seq(0, 45, 1)
ctd2 <- ctdDecimate(ctd, p = p)
lines(ctd2[["salinity"]], ctd2[["pressure"]], col = "blue")
p <- seq(0, 45, 1)
ctd3 <- ctdDecimate(ctd, p = p, method = function(x, y, xout) {
    predict(smooth.spline(x, y, df = 30), xout)$y
})
lines(ctd3[["salinity"]], ctd3[["pressure"]], col = "red")

}
\references{
\enumerate{
\item R.F. Reiniger and C.K. Ross, 1968.  A method of interpolation with
application to oceanographic data.  \emph{Deep Sea Research}, \strong{15},
185-193.
\item Oguma, Sachiko, Toru Suzuki, Yutaka Nagata, Hidetoshi Watanabe, Hatsuyo Yamaguchi,
and Kimio Hanawa. \dQuote{Interpolation Scheme for Standard Depth Data Applicable for Areas
with a Complex Hydrographical Structure.} Journal of Atmospheric and Oceanic Technology
21, no. 4 (April 1, 2004): 704-15.
}
}
\seealso{
The documentation for \linkS4class{ctd} explains the structure of
CTD objects, and also outlines the other functions dealing with them.

Other things related to ctd data: 
\code{\link{CTD_BCD2014666_008_1_DN.ODF.gz}},
\code{\link{[[,ctd-method}},
\code{\link{[[<-,ctd-method}},
\code{\link{as.ctd}()},
\code{\link{cnvName2oceName}()},
\code{\link{ctd}},
\code{\link{ctd-class}},
\code{\link{ctd.cnv.gz}},
\code{\link{ctdFindProfiles}()},
\code{\link{ctdFindProfilesRBR}()},
\code{\link{ctdRaw}},
\code{\link{ctdRepair}()},
\code{\link{ctdTrim}()},
\code{\link{ctd_aml.csv.gz}},
\code{\link{d200321-001.ctd.gz}},
\code{\link{d201211_0011.cnv.gz}},
\code{\link{handleFlags,ctd-method}},
\code{\link{initialize,ctd-method}},
\code{\link{initializeFlagScheme,ctd-method}},
\code{\link{oceNames2whpNames}()},
\code{\link{oceUnits2whpUnits}()},
\code{\link{plot,ctd-method}},
\code{\link{plotProfile}()},
\code{\link{plotScan}()},
\code{\link{plotTS}()},
\code{\link{read.ctd}()},
\code{\link{read.ctd.aml}()},
\code{\link{read.ctd.itp}()},
\code{\link{read.ctd.odf}()},
\code{\link{read.ctd.odv}()},
\code{\link{read.ctd.saiv}()},
\code{\link{read.ctd.sbe}()},
\code{\link{read.ctd.ssda}()},
\code{\link{read.ctd.woce}()},
\code{\link{read.ctd.woce.other}()},
\code{\link{setFlags,ctd-method}},
\code{\link{subset,ctd-method}},
\code{\link{summary,ctd-method}},
\code{\link{woceNames2oceNames}()},
\code{\link{woceUnit2oceUnit}()},
\code{\link{write.ctd}()}
}
\author{
Dan Kelley
}
\concept{things related to ctd data}
