% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/global.R
\name{diagThresh}
\alias{diagThresh}
\title{Estimation of the optimal threshold of a diagnostic marker}
\usage{
diagThresh(EvtDist = NULL, NoEvtDist = NULL, p, r, lowEvt = FALSE,
  le.MCMC = 1000, hessTol = 10^(-6), plot = FALSE,
  progress.bar = NULL, seed = NULL)
}
\arguments{
\item{EvtDist}{an object of class allowedFitDist that summarizes the distribution fitted to the marker values of patients with the disease of interest. This class of objects is obtained using the \code{\link[optimalThreshold]{fit}} function.}

\item{NoEvtDist}{an object of class allowedFitDist that summarizes the distribution fitted to the marker values of patients without the disease of interest. This class of objects is obtained using the \code{\link[optimalThreshold]{fit}} function.}

\item{p}{the prevalence of the disease in the target population.}

\item{r}{the risk threshold preference (see Details).}

\item{lowEvt}{logical argument that specifies whether low values of the marker are associated with the presence of the disease or not.}

\item{le.MCMC}{length of the desired MCMC chain.}

\item{hessTol}{a numeric value used in the optimization algorithm to control the tolerance threshold of the hessian value at the optimal threshold estimate.}

\item{plot}{a logical value indicating whether routine graphics must be produced.}

\item{progress.bar}{a character string indicating whether the user wishes to print a progress bar during the function process.}

\item{seed}{a numerical value used to fix the random seed.}
}
\value{
Returns an object of class \code{diagOptThresh}.
}
\description{
This function produces a sample of the posterior distribution of the optimal threshold of a diagnostic marker. The optimal threshold is defined as the marker value that maximized the utility of using the marker to make the diagnosis and treat the patient (treat or not the patient). The utility function takes into account the proportions of patients well classified and miss-classified (through the sensitivity and specificity), the prevalence of the disease in the target population, and the cost and benefits of treating wrongly or rightly the subject.
To calculate the utility function, the user needs to specify:
\itemize{
\item the distribution of the marker in the subject with and without the disease (see the \code{\link[optimalThreshold]{fit}} function)
\item the prevalence of the disease in the target population
\item the cost of treating subject without the disease and the benefit of treating a patient with the disease (see Details). 
}
The optimal threshold and its credible interval are calculated using a Monte Carlo approach.
}
\details{
The r value can be defined as the probability of disease above which a patient or a physician would accept the treatment. The value (1-r)/r can be interpreted as the NB/NC ratio, i.e. the number of subjects without the disease a physician would accept to treat wrongly to be able to detect and treat one diseased patient.
}
\section{References }{

Subtil, F, and Rabilloud. A Bayesian method to estimate the optimal threshold of a longitudinal marker. \emph{Biometrical Journal}. 2010; 52(3): 333-347.
}

\examples{
#Simulating data from two gaussian distributions:
xE <- rnorm(100) # distribution of the marker in diseased patients
xEb <- rnorm(400, 2) # distribution of the marker in the subjects without the disease

#When working with real data. You can check the decision rule (whether low or high 
#value of the marker are associated with the disease) using the densCurves function:
densCurves(x0 = xEb, x1 = xE, type = "diagnostic")

#Fit normal distributions on the two groups:
fitE <- fit(xE, "norm")
fitEb <- fit(xEb, "norm")

#Apply the main function to estimate the optimal threshold:
\donttest{
res <- diagThresh(fitE, fitEb, p = 0.2, r = 0.3, lowEvt = TRUE, le.MCMC = 5000, 
                  plot = TRUE, progress.bar = "text")

#You can summarize the results using the summary() function:
summary(res,method = "median")

#You can extract the estimates and CI bounds of each indicator presented in the summary:
estimates(res, method = "median")
credibleIntervals(res)

#Plot the decision curves (this function is time-consuming):
dCres <- decisionCurve(res, r = seq(0, 0.5, length.out = 10))

#You can plot again the decision curves by applying the plot method to dCres, 
#this function is a lot faster than the previous one. It also has more options
#to customize the plot:
plot(dCres)
}
}
