% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom-interpolation.r
\name{geom_interpolation}
\alias{geom_interpolation}
\title{Render interpolation of new rows from columns (or vice-versa)}
\usage{
geom_interpolation(
  mapping = NULL,
  data = NULL,
  stat = "identity",
  position = "identity",
  new_data = NULL,
  type = c("centroid", "sequence"),
  arrow = default_arrow,
  ...,
  point.fill = NA,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{stat}{The statistical transformation to use on the data for this layer.
When using a \verb{geom_*()} function to construct a layer, the \code{stat}
argument can be used the override the default coupling between geoms and
stats. The \code{stat} argument accepts the following:
\itemize{
\item A \code{Stat} ggproto subclass, for example \code{StatCount}.
\item A string naming the stat. To give the stat as a string, strip the
function name of the \code{stat_} prefix. For example, to use \code{stat_count()},
give the stat as \code{"count"}.
\item For more information and other ways to specify the stat, see the
\link[ggplot2:layer_stats]{layer stat} documentation.
}}

\item{position}{A position adjustment to use on the data for this layer. This
can be used in various ways, including to prevent overplotting and
improving the display. The \code{position} argument accepts the following:
\itemize{
\item The result of calling a position function, such as \code{position_jitter()}.
This method allows for passing extra arguments to the position.
\item A string naming the position adjustment. To give the position as a
string, strip the function name of the \code{position_} prefix. For example,
to use \code{position_jitter()}, give the position as \code{"jitter"}.
\item For more information and other ways to specify the position, see the
\link[ggplot2:layer_positions]{layer position} documentation.
}}

\item{new_data}{A list (best structured as a \link[base:data.frame]{data.frame})
of row (\code{geom_cols_interpolation()}) or column
(\code{geom_rows_interpolation()}) values to interpolate.}

\item{type}{Character value matched to \code{"centroid"} or \code{"sequence"}; the type
of operations used to visualize interpolation.}

\item{arrow}{Specification for arrows, as created by \code{\link[grid:arrow]{grid::arrow()}}, or else
\code{NULL} for no arrows.}

\item{...}{Additional arguments passed to \code{\link[ggplot2:layer]{ggplot2::layer()}}.}

\item{point.fill}{Default aesthetics for markers. Set to NULL to inherit from
the data's aesthetics.}

\item{na.rm}{Passed to \code{\link[ggplot2:layer]{ggplot2::layer()}}.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}
}
\description{
\code{geom_interpolation()} renders a geometric construction that
interpolates a new data matrix (row or column) element from its entries to
its artificial coordinates.
}
\details{
Interpolation answers the following question: Given a new data
element that might have appeared as a row (respectively, column) in the
singular-value-decomposed data matrix, where should we expect the marker
for this element to appear in the biplot? The solution is the vector sum of
the column (row) units weighted by their values in the new row (column).
Gower, Gardner--Lubbe, & le Roux (2011) provide two visualizations of this
calculation: a tail-to-head sequence of weighted units (\code{type = "sequence"}), and a centroid of the weighted units scaled by the number of
units (\code{type = "centroid"}).

\strong{WARNING:}
This layer is appropriate only with axes in standard coordinates (usually
\code{\link[=confer_inertia]{confer_inertia(p = "rows")}}) and interpolative
calibration (\code{\link[=ggbiplot]{ggbiplot(axis.type = "interpolative")}}).
}
\section{Biplot layers}{

\code{\link[=ggbiplot]{ggbiplot()}} uses \code{\link[ggplot2:fortify]{ggplot2::fortify()}} internally to produce a single data
frame with a \code{.matrix} column distinguishing the subjects (\code{"rows"}) and
variables (\code{"cols"}). The stat layers \code{stat_rows()} and \code{stat_cols()} simply
filter the data frame to one of these two.

The geom layers \verb{geom_rows_*()} and \verb{geom_cols_*()} call the corresponding
stat in order to render plot elements for the corresponding factor matrix.
\verb{geom_dims_*()} selects a default matrix based on common practice, e.g.
points for rows and arrows for columns.
}

\section{Aesthetics}{

\code{geom_interpolation()} requires the custom \strong{\code{interpolate}} aesthetic, which
tells the internals which columns of the \code{new_data} parameter contain the
variables to be used for interpolation. Except in rare cases, \code{new_data}
should contain the same rows or columns as the ordinated data and
\code{interpolate} should be set to \code{name} (procured by \code{\link[=augment_ord]{augment_ord()}}).
\code{geom_interpolation()} additionally understands the following aesthetics
(required aesthetics are in bold):
\itemize{
\item \code{alpha}
\item \code{colour}
\item \code{linetype}
\item \code{size}
\item \code{fill}
\item \code{shape}
\item \code{stroke}
\item \code{center}, \code{scale}
\item \code{group}
}
}

\examples{
iris[, -5] \%>\%
  prcomp(scale = TRUE) \%>\%
  as_tbl_ord() \%>\%
  print() -> iris_pca
iris_pca <- mutate_rows(iris_pca, species = iris$Species)
iris_pca <- augment_ord(iris_pca)

# sample of one of each species, with some missing measurements
new_data <- iris[c(42, 61, 110), seq(5, 1), drop = FALSE]
new_data[3L, "Sepal.Width"] <- NA
new_data[1L, "Petal.Length"] <- NA
print(new_data)

# centroid interpolation method
iris_pca \%>\%
  augment_ord() \%>\%
  mutate_rows(obs = dplyr::row_number()) \%>\%
  mutate_cols(measure = name) \%>\%
  ggbiplot() +
  theme_bw() +
  scale_color_brewer(type = "qual", palette = 2) +
  geom_origin(marker = "cross", alpha = .5) +
  geom_cols_interpolation(
    aes(center = center, scale = scale, interpolate = name), size = 3,
    new_data = new_data, type = "centroid", alpha = .5
  ) +
  geom_rows_text(aes(label = obs, color = species), alpha = .5, size = 3)

# missing an entire variable
new_data$Petal.Length <- NULL

# sequence interpolation method
iris_pca \%>\%
  augment_ord() \%>\%
  mutate_rows(obs = dplyr::row_number()) \%>\%
  mutate_cols(measure = name) \%>\%
  ggbiplot() +
  theme_bw() +
  scale_color_brewer(type = "qual", palette = 2) +
  geom_origin(marker = "circle", alpha = .5) +
  geom_cols_interpolation(
    aes(center = center, scale = scale, interpolate = name,
        linetype = measure),
    new_data = new_data, type = "sequence", alpha = .5
  ) +
  geom_rows_text(aes(label = obs, color = species), alpha = .5, size = 3)
}
\references{
Gower JC, Gardner--Lubbe S, & le Roux NJ (2011) \emph{Understanding Biplots}.
Wiley, ISBN: 978-0-470-01255-0. \url{https://www.wiley.com/go/biplots}
}
\seealso{
Other geom layers: 
\code{\link{geom_origin}()}
}
\concept{geom layers}
