% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fun-lda.r
\name{lda-ord}
\alias{lda-ord}
\alias{lda_ord}
\alias{lda_ord.formula}
\alias{lda_ord.data.frame}
\alias{lda_ord.matrix}
\alias{lda_ord.default}
\alias{predict.lda_ord}
\alias{model.frame.lda_ord}
\title{Augmented implementation of linear discriminant analysis}
\usage{
lda_ord(x, ...)

\method{lda_ord}{formula}(formula, data, ..., subset, na.action)

\method{lda_ord}{data.frame}(x, ...)

\method{lda_ord}{matrix}(x, grouping, ..., subset, na.action)

\method{lda_ord}{default}(
  x,
  grouping,
  prior = proportions,
  tol = 1e-04,
  method = c("moment", "mle", "mve", "t"),
  CV = FALSE,
  nu = 5,
  ...,
  ret.x = TRUE,
  ret.grouping = TRUE,
  axes.scale = "unstandardized"
)

\method{predict}{lda_ord}(
  object,
  newdata,
  prior = object$prior,
  dimen,
  method = c("plug-in", "predictive", "debiased"),
  ...
)

\method{model.frame}{lda_ord}(formula, ...)
}
\arguments{
\item{x}{
    (required if no formula is given as the principal argument.)
    a matrix or data frame or Matrix containing the explanatory variables.
  }

\item{...}{
    arguments passed to or from other methods.
}

\item{formula}{
    A formula of the form \code{groups ~ x1 + x2 + \dots}  That is, the
    response is the grouping factor and the right hand side specifies
    the (non-factor) discriminators.
  }

\item{data}{
    An optional data frame, list or environment from which variables
    specified in \code{formula} are preferentially to be taken.
  }

\item{subset}{
    An index vector specifying the cases to be used in the training
    sample.  (NOTE: If given, this argument must be named.)
  }

\item{na.action}{
    A function to specify the action to be taken if \code{NA}s are found.
    The default action is for the procedure to fail.  An alternative is
    \code{na.omit}, which leads to rejection of cases with missing values on
    any required variable.  (NOTE: If given, this argument must be named.)
  }

\item{grouping}{
    (required if no formula principal argument is given.)
    a factor specifying the class for each observation.
  }

\item{prior}{
    the prior probabilities of class membership.  If unspecified, the
    class proportions for the training set are used.  If present, the
    probabilities should be specified in the order of the factor
    levels.
  }

\item{tol}{
    A tolerance to decide if a matrix is singular; it will reject variables
    and linear combinations of unit-variance variables whose variance is
    less than \code{tol^2}.
  }

\item{method}{
    \code{"moment"} for standard estimators of the mean and variance,
    \code{"mle"} for MLEs, \code{"mve"} to use \code{\link[MASS]{cov.mve}}, or
    \code{"t"} for robust estimates based on a \eqn{t} distribution.
  }

\item{CV}{
    If true, returns results (classes and posterior probabilities) for
    leave-one-out cross-validation. Note that if the prior is estimated,
    the proportions in the whole dataset are used.
  }

\item{nu}{
    degrees of freedom for \code{method = "t"}.
  }

\item{ret.x, ret.grouping}{Logical; whether to retain as attributes the data
matrix (\code{x}) and the class assignments (\code{grouping}) on which LDA is
performed. Methods like \code{predict()} access these objects by name in the
parent environment, and retaining them as attributes prevents errors that
arise if these objects are reassigned.}

\item{axes.scale}{Character string indicating how to left-transform the
\code{scaling} value when rendering biplots using \code{\link[=ggbiplot]{ggbiplot()}}. Options include
\code{"unstandardized"}, \code{"standardized"}, and \code{"contribution"}.}

\item{object}{
object  of class \code{"lda"}
}

\item{newdata}{
  data frame of cases to be classified or, if \code{object}
  has a formula, a data frame with columns of the same names as the
  variables used.  A vector will be interpreted
  as a row vector.  If newdata is missing, an attempt will be
  made to retrieve the data used to fit the \code{lda} object.
}

\item{dimen}{
the dimension of the space to be used. If this is less than \code{min(p, ng-1)},
only the first \code{dimen} discriminant components are used (except for
\code{method="predictive"}), and only those dimensions are returned in \code{x}.
}
}
\value{
Output from \code{\link[MASS:lda]{MASS::lda()}} with an additional preceding \link{class}
'lda_ord' and up to three \link{attributes}:
\itemize{
\item the input data \code{x}, if \code{ret.x = TRUE}
\item the class assignments \code{grouping}, if \code{ret.grouping = TRUE}
\item if the parameter \code{axes.scale} is not 'unstandardized', a matrix
\code{axes.scale} that encodes the transformation of the row space
}
}
\description{
This function replicates \code{\link[MASS:lda]{MASS::lda()}} with options and defaults
to retain elements useful to the \link{tbl_ord} class and biplot calculations.
}
\details{
Linear discriminant analysis relies on an eigendecomposition of the product
\eqn{W^{-1}B} of the inverse of the within-class covariance matrix \eqn{W} by
the between-class covariance matrix \eqn{B}. This eigendecomposition can be
motivated as the right (\eqn{V}) half of the singular value decomposition of
the matrix of \emph{Mahalanobis distances} between the cases after "sphering"
(linearly transforming them so that the within-class covariance is the
identity matrix). LDA are not traditionally represented as biplots, with some
exceptions (Gardner & le Roux, 2005; Greenacre, 2010, p. 109--117).

LDA is implemented as \code{\link[MASS:lda]{MASS::lda()}} in the \strong{MASS} package, in which the
variables are transformed by a sphering matrix \eqn{S} (Venables & Ripley,
2003, p. 331--333). The returned element \code{scaling} contains the
unstandardized \emph{discriminant coefficients}, which define the discriminant
scores of the cases and their centroids as linear combinations of the
original variables.

The discriminant coefficients constitute one of several possible choices of
axes for a biplot representation of the LDA. The slightly modified function
\code{\link[=lda_ord]{lda_ord()}} provides additional options:
\itemize{
\item The \emph{standardized discriminant coefficients} are obtained by (re)scaling
the coefficients by the variable standard deviations. These coefficients
indicate the contributions of the variables to the discriminant scores after
controlling for their variances (Orlov, 2013).
\item The variables' \emph{contributions} to the Mahalanobis variance along each
discriminant axis are obtained by transforming the coefficients by the
inverse of the sphering matrix \eqn{S}. Because the contribution biplot
derives from the eigendecomposition of the Mahalanobis distance matrix, the
projections of the centroids and cases onto the variable axes approximate
their variable values after centering and sphering (Greenacre, 2013).
}

Finally, in contrast to \code{\link[MASS:lda]{MASS::lda()}}, \code{lda_ord()} defaults both \code{ret.x} and
\code{ret.grouping} to \code{TRUE}, so that these elements can be used to compute and
annotate case scores as \link[=supplementation]{supplementary} elements.
}
\examples{
# Anderson iris species data centroid
iris_centroid <- t(apply(iris[, 1:4], 2, mean))
# unstandardized discriminant coefficients: the discriminant axes are linear
# combinations of the centered variables
iris_lda <- lda_ord(iris[, 1:4], iris[, 5], axes.scale = "unstandardized")
# linear combinations of centered variables
print(sweep(iris_lda$means, 2, iris_centroid, "-") \%*\% get_cols(iris_lda))
# discriminant centroids
print(get_rows(iris_lda, elements = "active"))

# unstandardized coefficient LDA biplot
iris_lda \%>\%
  as_tbl_ord() \%>\%
  augment_ord() \%>\%
  ggbiplot() +
  theme_bw() +
  coord_scaffold() +
  geom_rows_point(aes(color = grouping), elements = "score", alpha = 1/3) +
  geom_rows_point(aes(color = grouping), size = 3) +
  geom_cols_vector(aes(label = name), color = "#888888", size = 3) +
  scale_color_brewer(type = "qual", palette = 2) +
  ggtitle("Unstandardized coefficient biplot of iris LDA")

# standardized discriminant coefficients: permit comparisons across the
# variables
iris_lda <- lda_ord(iris[, 1:4], iris[, 5], axes.scale = "standardized")
# standardized variable contributions to discriminant axes
iris_lda \%>\%
  as_tbl_ord() \%>\%
  augment_ord() \%>\%
  fortify(.matrix = "cols") \%>\%
  dplyr::mutate(variable = name) \%>\%
  tidyr::gather(discriminant, coefficient, LD1, LD2) \%>\%
  ggplot(aes(x = discriminant, y = coefficient, fill = variable)) +
  geom_bar(position = "dodge", stat = "identity") +
  labs(y = "Standardized coefficient", x = "Linear discriminant") +
  theme_bw() +
  coord_flip()
# standardized coefficient LDA biplot
iris_lda \%>\%
  as_tbl_ord() \%>\%
  augment_ord() \%>\%
  ggbiplot() +
  theme_bw() +
  coord_scaffold() +
  geom_rows_point(aes(color = grouping), elements = "score", alpha = 1/3) +
  geom_rows_point(aes(color = grouping), size = 3) +
  geom_cols_vector(aes(label = name), color = "#888888", size = 3) +
  scale_color_brewer(type = "qual", palette = 2) +
  ggtitle("Standardized coefficient biplot of iris LDA")

# variable contributions (de-sphered discriminant coefficients): recover the
# inner product relationship with the centered class centroids
iris_lda <- lda_ord(iris[, 1:4], iris[, 5], axes.scale = "contribution")
# symmetric square root of within-class covariance
C_W_eig <- eigen(cov(iris[, 1:4] - iris_lda$means[iris[, 5], ]))
C_W_sqrtinv <-
  C_W_eig$vectors \%*\% diag(1/sqrt(C_W_eig$values)) \%*\% t(C_W_eig$vectors)
# product of matrix factors (scores and loadings)
print(get_rows(iris_lda, elements = "active") \%*\% t(get_cols(iris_lda)))
# "asymmetric" square roots of Mahalanobis distances between variables
print(sweep(iris_lda$means, 2, iris_centroid, "-") \%*\% C_W_sqrtinv)
# contribution LDA biplot
iris_lda \%>\%
  as_tbl_ord() \%>\%
  augment_ord() \%>\%
  ggbiplot() +
  theme_bw() +
  coord_scaffold() +
  geom_rows_point(aes(color = grouping), elements = "score", alpha = 1/3) +
  geom_rows_point(aes(color = grouping), size = 3) +
  geom_cols_vector(aes(label = name), color = "#888888", size = 3) +
  scale_color_brewer(type = "qual", palette = 2) +
  ggtitle("Contribution biplot of iris LDA")
}
\references{
Gardner S & le Roux NJ (2005) "Extensions of Biplot Methodology to
Discriminant Analysis". \emph{Journal of Classification} \strong{22}(1): 59--86.
\doi{10.1007/s00357-005-0006-7}
\url{https://link.springer.com/article/10.1007/s00357-005-0006-7}

Greenacre MJ (2010) \emph{Biplots in Practice}. Fundacion BBVA, ISBN:
978-84-923846.
\url{https://www.fbbva.es/microsite/multivariate-statistics/biplots.html}

Venables WN & Ripley BD (2003) \emph{Modern Applied Statistics with S}, Fourth
Edition. Springer Science & Business Media, ISBN: 0387954570, 9780387954578.
\url{https://www.mimuw.edu.pl/~pokar/StatystykaMgr/Books/VenablesRipley_ModernAppliedStatisticsS02.pdf}

Orlov K (2013) \emph{Answer to} "Algebra of LDA. Fisher discrimination power of a
variable and Linear Discriminant Analysis". CrossValidated, accessed
2019-07-26. \url{https://stats.stackexchange.com/a/83114/68743}

Greenacre M (2013) "Contribution Biplots". \emph{Journal of Computational and
Graphical Statistics}, \strong{22}(1): 107--122.
\url{https://www.tandfonline.com/doi/abs/10.1080/10618600.2012.702494}
}
\seealso{
\code{\link[MASS:lda]{MASS::lda()}}, from which \code{lda_ord()} is adapted
}
