% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grid_search_cv.R
\name{grid_search_cv}
\alias{grid_search_cv}
\title{Cross-validated grid search for PCP models}
\usage{
grid_search_cv(
  D,
  pcp_fn,
  grid,
  ...,
  parallel_strategy = "sequential",
  num_workers = 1,
  perc_test = 0.05,
  num_runs = 100,
  return_all_tests = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{D}{The input data matrix (can contain \code{NA} values). Note that PCP will
converge much more quickly when \code{D} has been standardized in some way (e.g.
scaling columns by their standard deviations, or column-wise min-max
normalization).}

\item{pcp_fn}{The PCP function to use when grid searching. Must be either
\code{rrmc} or \code{root_pcp} (passed without the soft brackets).}

\item{grid}{A \code{data.frame} of dimension \code{j} by \code{k} containing the \code{j}-many
unique settings of \code{k}-many parameters to try.
\strong{NOTE: The columns of \code{grid} should be
named after the required parameters in the function header of \code{pcp_fn}.}
For example, if \code{pcp_fn = root_pcp} and you want to search through \code{lambda}
and \code{mu}, then \code{names(grid)} must be set to \code{c("lambda", "mu")}. If instead
you want to keep e.g. \code{lambda} fixed and search through only \code{mu}, you can
either have a \code{grid} with only one column, \code{mu}, and pass \code{lambda} as a
constant via \code{...}, or you can have \code{names(grid)} set to
\code{c("lambda", "mu")} where \code{lambda} is constant. The same logic applies for
\code{pcp_fn = rrmc} and \code{eta} and \code{r}.}

\item{...}{Any parameters required by \code{pcp_fn} that should be kept constant
throughout the grid search, or those parameters that cannot be stored in
\code{grid} (e.g. the \code{LOD} parameter). A parameter should not be passed with
\code{...} if it is already a column in \code{grid}, as that behavior is ambiguous.}

\item{parallel_strategy}{(Optional) The parallelization strategy used when
conducting the grid search (to be passed on to the \code{\link[future:plan]{future::plan()}}
function). Must be one of: \code{"sequential"}, \code{"multisession"}, \code{"multicore"}
or \code{"cluster"}. By default, \code{parallel_strategy = "sequential"}, which
runs the grid search in serial and the \code{num_workers} argument is ignored.
The option \code{parallel_strategy = "multisession"} parallelizes the search
via sockets in separate R \emph{sessions}. The option
\code{parallel_strategy = "multicore"} is not supported on Windows
machines, nor in .Rmd files (must be run in a .R script) but parallelizes
the search much faster than \code{"multisession"} since it runs separate
\emph{forked} R processes. The option \code{parallel_strategy = "cluster"}
parallelizes using separate R sessions running typically on one or more
machines. Support for other parallel strategies will be added in a future
release of \code{pcpr}. \strong{It is recommended to use
\code{parallel_strategy = "multicore"} or \code{"multisession"} when possible.}}

\item{num_workers}{(Optional) An integer specifying the number of workers to
use when parallelizing the grid search, to be passed on to
\code{\link[future:plan]{future::plan()}}. By default, \code{num_workers = 1}. When possible, it is
recommended to use \code{num_workers = parallel::detectCores(logical = F)},
which computes the number of physical CPUs available on the machine
(see \code{\link[parallel:detectCores]{parallel::detectCores()}}). \code{num_workers} is ignored
when \code{parallel_strategy = "sequential"}, and must be \verb{> 1} otherwise.}

\item{perc_test}{(Optional) The fraction of entries of \code{D} that will be
randomly corrupted as \code{NA} missing values (the test set). Can be anthing in
the range \verb{[0, 1)}. By default, \code{perc_test = 0.05}. See \strong{Best practices}
section for more details.}

\item{num_runs}{(Optional) The number of times to test a given parameter
setting. By default, \code{num_runs = 100}. See \strong{Best practices} section for
more details.}

\item{return_all_tests}{(Optional) A logical indicating if you would like the
output from all the calls made to \code{pcp_fn} over the course of the grid
search to be returned to you in list format. If set to \code{FALSE}, then only
statistics on the parameters tested will be returned. If set to \code{TRUE} then
every \code{L}, and \code{S} matrix recovered during the grid search will be returned
in the lists \code{L_mats} and \code{S_mats}, every test set matrix will be returned
in the list \code{test_mats}, the original input matrix will be returned as
\code{original_mat}, and the parameters passed in to \code{...} will be returned in
the \code{constant_params} list. \strong{By default, \code{return_all_tests = FALSE},
which is highly recommended. Setting \code{return_all_tests = TRUE} can consume
a massive amount of memory depending on the size of \code{grid}, the input
matrix \code{D}, and the value for \code{num_runs}.}}

\item{verbose}{(Optional) A logical indicating if you would like verbose
output displayed or not. By default, \code{verbose = TRUE}. To obtain
progress bar updates, the user must wrap the \code{grid_search_cv()} call
with a call to \code{\link[progressr:with_progress]{progressr::with_progress()}}. The progress bar does \emph{not}
depend on the value passed for \code{verbose}.}
}
\value{
A list containing:
\itemize{
\item \code{all_stats}: A \code{data.frame} containing the statistics of every run
comprising the grid search. These statistics include the parameter
settings for the run, along with the \code{run_num} (used as the seed
for the corruption step, step 1 in the grid search procedure),
the relative error for the run \code{rel_err}, the rank of the recovered L
matrix \code{L_rank}, the sparsity of the recovered S matrix \code{S_sparsity},
the number of \code{iterations} PCP took to reach convergence (for \code{\link[=root_pcp]{root_pcp()}}
only), and the error status \code{run_error} of the PCP run (\code{NA} if no error,
otherwise a character string).
\item \code{summary_stats}: A \code{data.frame} containing a summary of the information in
\code{all_stats}. Summary made by column-wise averaging the results in
\code{all_stats}.
\item \code{metadata}: A character string containing the metadata associated with the
grid search instance.
}

If \code{return_all_tests = TRUE} then the following are also returned as part
of the list:
\itemize{
\item \code{L_mats}: A list containing all the \code{L} matrices returned from PCP
throughout the grid search. Therefore, \code{length(L_mats) == nrow(all_stats)}.
Row \code{i} in \code{all_stats} corresponds to \code{L_mats[[i]]}.
\item \code{S_mats}: A list containing all the S matrices returned from PCP throughout
the grid search. Therefore, \code{length(S_mats) == nrow(all_stats)}. Row \code{i} in
\code{all_stats} corresponds to \code{S_mats[[i]]}.
\item \code{test_mats}: A list of \code{length(num_runs)} containing all the corrupted test
mats (and their masks) used throughout the grid search. Note:
\code{all_stats$run[i]} corresponds to \code{test_mats[[i]]}.
\item \code{original_mat}: The original data matrix \code{D}.
\item \code{constant_params}: A copy of the constant parameters that were originally
passed to the grid search (for record keeping).
}
}
\description{
\code{grid_search_cv()} conducts a Monte Carlo style cross-validated grid search
of PCP parameters for a given data matrix \code{D}, PCP function \code{pcp_fn}, and
grid of parameter settings to search through \code{grid}. The run time of the grid
search can be sped up using bespoke parallelization settings. The call to
\code{grid_search_cv()} can be wrapped in a call to \code{\link[progressr:with_progress]{progressr::with_progress()}}
for progress bar updates. See the below sections for details.
}
\section{The Monte Carlo style cross-validation procedure}{

Each hyperparameter setting is cross-validated by:
\enumerate{
\item Randomly corrupting \code{perc_test} percent of the entries in \code{D} as missing
(i.e. \code{NA} values), yielding \code{D_tilde}. Done via \code{\link[=sim_na]{sim_na()}}.
\item Running the PCP function \code{pcp_fn} on \code{D_tilde}, yielding estimates \code{L}
and \code{S}.
\item Recording the relative recovery error of \code{L} compared with the input data
matrix \code{D} for \emph{only those values that were imputed as missing during the
corruption step} (step 1 above). Mathematically, calculate:
\eqn{||P_{\Omega^c}(D - L)||_F / ||P_{\Omega^c}(D)||_F}, where
\eqn{P_{\Omega^c}} selects only those entries where
\code{is.na(D_tilde) == TRUE}.
\item Repeating steps 1-3 for a total of \code{num_runs}-many times, where each "run"
has a unique random seed from \code{1} to \code{num_runs} associated with it.
\item Performance statistics can then be calculated for each "run", and then
summarized across all runs for average model performance statistics.
}
}

\section{Best practices for \code{perc_test} and \code{num_runs}}{

Experimentally, this grid search procedure retrieves the best performing
PCP parameter settings when \code{perc_test} is relatively low, e.g.
\code{perc_test = 0.05}, or 5\%, and \code{num_runs} is relatively high, e.g.
\code{num_runs = 100}.

The larger \code{perc_test} is, the more the test set turns into a matrix
completion problem, rather than the desired matrix decomposition problem. To
better resemble the actual problem PCP will be faced with come inference
time, \code{perc_test} should therefore be kept relatively low.

Choosing a reasonable value for \code{num_runs} is dependent on the need to keep
\code{perc_test} relatively low. Ideally, a large enough \code{num_runs} is used so
that many (if not all) of the entries in \code{D} are likely to eventually be
tested. Note that since test set entries are chosen randomly for all runs \code{1}
through \code{num_runs}, in the pathologically worst case scenario, the same exact
test set could be drawn each time. In the best case scenario, a different
test set is obtained each run, providing balanced coverage of \code{D}. Viewed
another way, the smaller \code{num_runs} is, the more the results are susceptible
to overfitting to the relatively few selected test sets.
}

\section{Interpretaion of results}{

Once the grid search of has been conducted, the optimal hyperparameters can
be chosen by examining the output statistics \code{summary_stats}. Below are a
few suggestions for how to interpret the \code{summary_stats} table:
\itemize{
\item Generally speaking, the first thing a user will want to inspect is the
\code{rel_err} statistic, capturing the relative discrepancy between recovered
test sets and their original, observed (yet possibly noisy) values. Lower
\code{rel_err} means the PCP model was better able to recover the held-out test
set. So, in general, \strong{the best parameter settings are those with the
lowest \code{rel_err}.} Having said this, it is important to remember that this
statistic should be taken with a grain of salt: Because no ground truth \code{L}
matrix exists, the \code{rel_err} measurement is forced to rely on the
comparison between the \emph{noisy observed data} matrix \code{D} and the \emph{estimated
low-rank model} \code{L}. So the \code{rel_err} metric is an "apples to oranges"
relative error. For data that is a priori expected to be subject to a
high degree of noise, it may actually be better to \emph{discard}
parameter settings with \emph{suspiciously low} \code{rel_err}s (in which
case the solution may be hallucinating an inaccurate low-rank structure
from the observed noise).
\item For grid searches using \code{\link[=root_pcp]{root_pcp()}} as the PCP model, parameters that
fail to converge can be discarded. Generally, fewer \code{\link[=root_pcp]{root_pcp()}} iterations
(\code{num_iter}) taken to reach convergence portend a more reliable / stable
solution. In rare cases, the user may need to increase \code{\link[=root_pcp]{root_pcp()}}'s
\code{max_iter} argument to reach convergence. \code{\link[=rrmc]{rrmc()}} does not report
convergence metadata, as its optimization scheme runs for a fixed
number of iterations.
\item Parameter settings with unreasonable sparsity or rank measurements
can also be discarded. Here, "unreasonable" means these reported metrics
flagrantly contradict prior assumptions, knowledge, or work. For instance,
most air pollution datasets contain a number of extreme exposure events, so
PCP solutions returning sparse \code{S} models with 100\% sparsity have obviously
been regularized too heavily. Solutions with lower sparsities should be
preferred. Note that reported sparsity and rank measurements are \emph{estimates
heavily dependent on the \code{thresh} set by the \code{\link[=sparsity]{sparsity()}} & \code{\link[=matrix_rank]{matrix_rank()}}
functions}. E.g. it could be that the actual average matrix rank is much
higher or lower when a threshold that better takes into account the
relative scale of the singular values is used. Likewise for the sparsity
estimations. Also, recall that the given value for \code{perc_test} artifically
sets a sparsity floor, since those missing entries in the test set cannot
be recovered in the \code{S} matrix. E.g. if \code{perc_test = 0.05}, then no
parameter setting will have an estimated sparsity lower than 5\%.
}
}

\examples{
#### -------Simple simulated PCP problem-------####
# First we will simulate a simple dataset with the sim_data() function.
# The dataset will be a 100x10 matrix comprised of:
# 1. A rank-3 component as the ground truth L matrix;
# 2. A ground truth sparse component S w/outliers along the diagonal; and
# 3. A dense Gaussian noise component
data <- sim_data()
#### -------Tiny grid search-------####
# Here is a tiny grid search just to test the function quickly.
# In practice we would recommend a larger grid search.
# For examples of larger searches, see the vignettes.
gs <- grid_search_cv(
  data$D,
  rrmc,
  data.frame("eta" = 0.35),
  r = 3,
  num_runs = 2
)
gs$summary_stats
}
\seealso{
\code{\link[=sim_na]{sim_na()}}, \code{\link[=sparsity]{sparsity()}}, \code{\link[=matrix_rank]{matrix_rank()}}, \code{\link[=get_pcp_defaults]{get_pcp_defaults()}}
}
