#' get_request_token
#' @description Requests a request_token for the Pocket application with a given consumer key from the Pocket Authentication API. The request token can then be used in \code{\link{create_authorize_url}} and \code{\link{get_access_token}}.
#' @param consumer_key Character string. Here goes your Pocket consumer key.
#' @return Character string. Request token for the Pocket Application corresponding to your consumer_key.
#' @details See the \href{https://github.com/CorrelAid/pocketapi}{GitHub README} for details on Authentication.
#' @importFrom httr POST stop_for_status content
#' @family authentication functions
#' @export
get_request_token <- function(consumer_key) {
  if (!is.character(consumer_key) || length(consumer_key) != 1) {
    usethis::ui_stop("Argument consumer_key must be a character vector of length 1.")
  }

  # Define relevant URLs for the function
  REQUEST_URL <- "https://getpocket.com/v3/oauth/request"
  REDIRECT_URI <- "https://github.com/CorrelAid/pocketapi"

  # Get request_token
  res <- httr::POST(REQUEST_URL,
    body = list(consumer_key = consumer_key, redirect_uri = REDIRECT_URI), encode = "form"
  )
  pocket_stop_for_status_(res)

  request_token <- httr::content(res)$code
  return(request_token)
}

#' create_authorize_url
#' @description Creates the URL the user needs to enter into her browser in order to authorize the Pocket application/the request token.
#' @param request_token Character string. Pocket request token generated by the function \code{\link{get_request_token}}.
#' @details See the \href{https://github.com/CorrelAid/pocketapi}{GitHub README} for details on Authentication.
#' @return Character string. The URL for authorizing the Pocket application for which the request token was requested. This is what you enter into your browser.
#' @importFrom glue glue
#' @family authentication functions
#' @export
#'
create_authorize_url <- function(request_token) {
  # Only accept a string
  if (!is.character(request_token) || length(request_token) != 1) {
    usethis::ui_stop("Argument request_token must be a character vector of length 1.")
  }
  # Create URL to give the app access
  AUTHORIZE_URL <- "https://getpocket.com/auth/authorize"
  REDIRECT_URI <- "https://github.com/CorrelAid/pocketapi"

  auth_url <- glue::glue("{AUTHORIZE_URL}?request_token={request_token}&redirect_uri={REDIRECT_URI}")

  usethis::ui_todo("Enter this URL into your browser and grant your app access:")
  return(auth_url)
}

#' get_access_token
#' @description Creates the URL the user needs to enter into her browser in order to authorize the Pocket application/the request token.
#' @param consumer_key Character string. Here goes your Pocket consumer key.
#' @param request_token Character string. Pocket request token generated by \code{\link{get_request_token}}.
#' @return Character string. Returns the access token.
#' @details See the \href{https://github.com/CorrelAid/pocketapi}{GitHub README} for details on Authentication.
#' @importFrom httr POST stop_for_status content parse_url
#' @family authentication functions
#' @export
#'
get_access_token <- function(consumer_key, request_token) {
  if (!is.character(consumer_key) || length(consumer_key) != 1) {
    usethis::ui_stop("Argument consumer_key must be a character vector of length 1.")
  }

  if (!is.character(request_token) || length(request_token) != 1) {
    usethis::ui_stop("Argument request_token must be a character vector of length 1.")
  }

  AUTH_URL <- "https://getpocket.com/v3/oauth/authorize"

  authorize_url <- httr::parse_url(AUTH_URL)
  res <- httr::POST(authorize_url, body = list(consumer_key = consumer_key, code = request_token))

  pocket_stop_for_status_(res)
  access_token <- httr::content(res)$access_token
  return(access_token)
}
