% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CATE_surv.R
\name{catefitsurv}
\alias{catefitsurv}
\title{Estimation of the conditional average treatment effect (CATE) score for survival data}
\usage{
catefitsurv(
  data,
  score.method,
  cate.model,
  ps.model,
  ps.method = "glm",
  initial.predictor.method = "randomForest",
  ipcw.model = NULL,
  ipcw.method = "breslow",
  minPS = 0.01,
  maxPS = 0.99,
  followup.time = NULL,
  tau0 = NULL,
  higher.y = TRUE,
  prop.cutoff = seq(0.5, 1, length = 6),
  surv.min = 0.025,
  tree.depth = 2,
  n.trees.rf = 1000,
  n.trees.boosting = 200,
  B = 3,
  Kfold = 5,
  plot.gbmperf = TRUE,
  error.maxNR = 0.001,
  max.iterNR = 100,
  tune = c(0.5, 2),
  seed = NULL,
  verbose = 0,
  ...
)
}
\arguments{
\item{data}{A data frame containing the variables in the outcome, propensity score, and inverse
probability of censoring models (if specified); a data frame with \code{n} rows (1 row per observation).}

\item{score.method}{A vector of one or multiple methods to estimate the CATE score.
Allowed values are: \code{'randomForest'}, \code{'boosting'}, \code{'poisson'}, \code{'twoReg'}, and
\code{'contrastReg'}.}

\item{cate.model}{A standard \code{Surv} formula describing the outcome model to be fitted.
The outcome must appear on the left-hand side.}

\item{ps.model}{A formula describing the propensity score (PS) model to be fitted. The treatment must
appear on the left-hand side. The treatment must be a numeric vector coded as 0/1.
If data are from a randomized controlled trial, specify \code{ps.model = ~1} as an intercept-only model.}

\item{ps.method}{A character value for the method to estimate the propensity score.
Allowed values include one of:
\code{'glm'} for logistic regression with main effects only (default), or
\code{'lasso'} for a logistic regression with main effects and LASSO penalization on
two-way interactions (added to the model if interactions are not specified in \code{ps.model}).
Relevant only when \code{ps.model} has more than one variable.}

\item{initial.predictor.method}{A character vector for the method used to get initial
outcome predictions conditional on the covariates specified in \code{cate.model}. Only applies
when \code{score.method} includes \code{'twoReg'} or \code{'contrastReg'}. Allowed values include
one of \code{'randomForest'}, \code{'boosting'} and \code{'logistic'} (fastest).
Default is \code{'randomForest'}.}

\item{ipcw.model}{A formula describing the inverse probability of censoring weighting (IPCW)
model to be fitted. The left-hand side must be empty. Default is \code{ipcw.model = NULL},
which corresponds to specifying the IPCW model with the same covariates as the outcome model
\code{cate.model} plus the treatment.}

\item{ipcw.method}{A character value for the censoring model. Allowed values are:
\code{'breslow'} (Cox regression with Breslow estimator of the baseline survivor function),
\code{'aft (exponential)'}, \code{'aft (weibull)'}, \code{'aft (lognormal)'}
or \code{'aft (loglogistic)'} (accelerated failure time model with different distributions for
y variable). Default is \code{'breslow'}.}

\item{minPS}{A numerical value (in `[0, 1]`) below which estimated propensity scores should be
truncated. Default is \code{0.01}.}

\item{maxPS}{A numerical value (in `(0, 1]`) above which estimated propensity scores should be
truncated. Must be strictly greater than \code{minPS}. Default is \code{0.99}.}

\item{followup.time}{A column name in \code{data} specifying the maximum follow-up time,
interpreted as the potential censoring time. Default is \code{followup.time = NULL},
which corresponds to unknown potential censoring time.}

\item{tau0}{The truncation time for defining restricted mean time lost. Default is \code{NULL},
which corresponds to setting the truncation time as the maximum survival time in the data.}

\item{higher.y}{A logical value indicating whether higher (\code{TRUE}) or
lower (\code{FALSE}) values of the outcome are more desirable. Default is \code{TRUE}.}

\item{prop.cutoff}{A vector of numerical values (in `(0, 1]`) specifying percentiles of the
estimated log CATE scores to define nested subgroups. Each element represents the cutoff to
separate observations in nested subgroups (below vs above cutoff).
The length of \code{prop.cutoff} is the number of nested subgroups.
An equally-spaced sequence of proportions ending with 1 is recommended.
Default is \code{seq(0.5, 1, length = 6)}.}

\item{surv.min}{Lower truncation limit for the probability of being censored.
It must be a positive value and should be chosen close to 0. Default is \code{0.025}.}

\item{tree.depth}{A positive integer specifying the depth of individual trees in boosting
(usually 2-3). Used only if \code{score.method = 'boosting'} or
if \code{initial.predictor.method = 'boosting'} with \code{score.method = 'twoReg'} or
\code{'contrastReg'}. Default is 2.}

\item{n.trees.rf}{A positive integer specifying the maximum number of trees in random forest.
Used if \code{score.method = 'ranfomForest'} or if \code{initial.predictor.method = 'randomForest'}
with \code{score.method = 'twoReg'} or \code{'contrastReg'}. Only applies for survival outcomes.
Default is \code{1000}.}

\item{n.trees.boosting}{A positive integer specifying the maximum number of trees in boosting
(usually 100-1000). Used if \code{score.method = 'boosting'} or
if \code{initial.predictor.method = 'boosting'} with \code{score.method = 'twoReg'} or
\code{'contrastReg'}. Default is \code{200}.}

\item{B}{A positive integer specifying the number of time cross-fitting is repeated in
\code{score.method = 'twoReg'} and \code{'contrastReg'}. Default is \code{3}.}

\item{Kfold}{A positive integer specifying the number of folds used in cross-fitting
to partition the data in \code{score.method = 'twoReg'} and \code{'contrastReg'}.
Default is \code{5}.}

\item{plot.gbmperf}{A logical value indicating whether to plot the performance measures in
boosting. Used only if \code{score.method = 'boosting'} or if \code{score.method = 'twoReg'}
or \code{'contrastReg'} and \code{initial.predictor.method = 'boosting'}. Default is \code{TRUE}.}

\item{error.maxNR}{A numerical value > 0 indicating the minimum value of the mean absolute
error in Newton Raphson algorithm. Used only if \code{score.method = 'contrastReg'}.
Default is \code{0.001}.}

\item{max.iterNR}{A positive integer indicating the maximum number of iterations in the
Newton Raphson algorithm. Used only if \code{score.method = 'contrastReg'}.
Default is \code{150}.}

\item{tune}{A vector of 2 numerical values > 0 specifying tuning parameters for the
Newton Raphson algorithm. \code{tune[1]} is the step size, \code{tune[2]} specifies a
quantity to be added to diagonal of the slope matrix to prevent singularity.
Used only if \code{score.method = 'contrastReg'}. Default is \code{c(0.5, 2)}.}

\item{seed}{An optional integer specifying an initial randomization seed for reproducibility.
Default is \code{NULL}, corresponding to no seed.}

\item{verbose}{An integer value indicating what kind of intermediate progress messages should
be printed. \code{0} means no outputs. \code{1} means only progress and run time.
\code{2} means progress, run time, and all errors and warnings. Default is \code{0}.}

\item{...}{Additional arguments for \code{gbm()}}
}
\value{
Returns an object of the class \code{catefit} containing the following components:
\itemize{
 \item{\code{ate.randomForest}: } A vector of numerical values of length \code{prop.cutoff}
 containing the estimated ATE by the RMTL ratio in nested subgroups (defined by \code{prop.cutoff})
 constructed based on the estimated CATE scores with random forest method.
 Only provided if \code{score.method} includes \code{'randomForest'}.
 \item{\code{ate.boosting}: }Same as \code{ate.randomForest}, but with the nested subgroups based
 the estimated CATE scores with boosting. Only provided if \code{score.method}
 includes \code{'boosting'}.
 \item{\code{ate.poisson}: }Same as \code{ate.randomForest}, but with the nested subgroups based
 the estimated CATE scores with poisson regression.
 Only provided if \code{score.method} includes \code{'poisson'}.
 \item{\code{ate.twoReg}: }Same as \code{ate.randomForest}, but with the nested subgroups based
 the estimated CATE scores with two regressions.
 Only provided if \code{score.method} includes \code{'twoReg'}.
 \item{\code{ate.contrastReg}: }Same as \code{ate.randomForest}, but with the nested subgroups based
 the estimated CATE scores with contrast regression.
 Only provided if \code{score.method} includes \code{'contrastReg'}.
 \item{\code{hr.randomForest}: }A vector of numerical values of length \code{prop.cutoff}
 containing the adjusted hazard ratio in nested subgroups (defined by \code{prop.cutoff})
 constructed based on the estimated CATE scores with random forest method.
 Only provided if \code{score.method} includes \code{'randomForest'}.
 \item{\code{hr.boosting}: }Same as \code{hr.randomForest}, but with the nested subgroups based
 the estimated CATE scores with boosting. Only provided if \code{score.method}
 includes \code{'boosting'}.
 \item{\code{hr.poisson}: }Same as \code{hr.randomForest}, but with the nested subgroups based
 the estimated CATE scores with poisson regression.
 Only provided if \code{score.method} includes \code{'poisson'}.
 \item{\code{hr.twoReg}: }Same as \code{hr.randomForest}, but with the nested subgroups based
 the estimated CATE scores with two regressions.
 Only provided if \code{score.method} includes \code{'twoReg'}.
 \item{\code{hr.contrastReg}: }Same as \code{hr.randomForest}, but with the nested subgroups based
 the estimated CATE scores with contrast regression.
 Only provided if \code{score.method} includes \code{'contrastReg'}.
 \item{\code{score.randomForest}: }A vector of numerical values of length n
 (number of observations in \code{data}) containing the estimated log-CATE scores
 according to random forest. Only provided if \code{score.method}
 includes \code{'randomForest'}.
 \item{\code{score.boosting}: }Same as \code{score.randomForest}, but with estimated log-CATE score
 according to boosting. Only provided if \code{score.method} includes
 \code{'boosting'}.
 \item{\code{score.poisson}: }Same as \code{score.randomForest}, but with estimated log-CATE score
 according to the Poisson regression. Only provided if \code{score.method}
 includes \code{'poisson'}.
 \item{\code{score.twoReg}: }Same as \code{score.randomForest}, but with estimated log-CATE score
 according to two regressions. Only provided if \code{score.method} includes
 \code{'twoReg'}.
 \item{\code{score.contrastReg}: }Same as \code{score.randomForest}, but with estimated log-CATE score
 according to contrast regression. Only provided if \code{score.method} includes
 \code{'contrastReg'}.
 \item{\code{fit}: }Additional details on model fitting if \code{score.method}
 includes 'randomForest', 'boosting' or 'contrastReg':
 \itemize{
   \item{\code{result.randomForest}: }Details on the random forest model fitted to observations
   with treatment = 0 \code{($fit0.rf)} and to observations with treatment = 1 \code{($fit1.rf)}.
   Only provided if \code{score.method} includes \code{'randomForest'}.
   \item{\code{result.boosting}: }Details on the boosting model fitted to observations
   with treatment = 0, \code{($fit0.boosting)} and \code{($fit0.gam)} and to observations with treatment = 1,
   \code{($fit1.boosting)} and \code{($fit1.gam)}.
   Only provided if \code{score.method} includes \code{'boosting'}.
   \item{\code{result.contrastReg$converge.contrastReg}: }Whether the contrast regression algorithm converged
   or not. Only provided if \code{score.method} includes \code{'contrastReg'}.
 }
 \item{\code{coefficients}: }A data frame with the coefficients of the estimated log-CATE
 score by \code{score.method}. The data frame has number of rows equal to the number of
 covariates in \code{cate.model} and number of columns equal to \code{length(score.method)}.
 If \code{score.method} includes \code{'contrastReg'}, the data frame has an additional
 column containing the standard errors of the coefficients estimated with contrast regression.
 \code{'randomForest'} and \code{'boosting'} do not have coefficient results because
 tree-based methods typically do not express the log-CATE as a linear combination of coefficients
 and covariates.
 \item{\code{errors/warnings}: }A nested list of errors and warnings that were wrapped during the
 calculation of ATE. Errors and warnings are organized by \code{score.method}.
}
}
\description{
Provides singly robust and doubly robust estimation of CATE score for survival data with
up to 5 scoring methods among the following: Random forest, boosting, poisson regression,
two regressions, and contrast regression.
}
\details{
The CATE score represents an individual-level treatment effect for survival data,
estimated with random forest, boosting, Poisson regression, and the doubly
robust estimator (two regressions, Yadlowsky, 2020) applied separately by treatment group
or with the other doubly robust estimators (contrast regression, Yadlowsky, 2020) applied
to the entire data set.

\code{\link{catefitsurv}()} provides the coefficients of the CATE score for each scoring method requested
through \code{score.method}. Currently, contrast regression is the only method which allows
for inference of the CATE coefficients by providing standard errors of the coefficients.
The coefficients can be used to learn the effect size of each variable and predict the
CATE score for a new observation.

\code{\link{catefitsurv}()} also provides the predicted CATE score of each observation in the data set,
for each scoring method. The predictions allow ranking the observations from potentially
high responders to the treatment to potentially low or standard responders.

The estimated ATE among nested subgroups of high responders are also provided by scoring method.
Note that the ATEs in \code{\link{catefitsurv}()} are derived based on the CATE score which is estimated
using the same data sample. Therefore, overfitting may be an issue. \code{\link{catecvsurv}()} is more
suitable to inspect the estimated ATEs across scoring methods as it implements internal cross
validation to reduce optimism.
}
\examples{
\donttest{
library(survival)

tau0 <- with(survivalExample, min(quantile(y[trt == "drug1"], 0.95),
                               quantile(y[trt == "drug0"], 0.95)))

fit <- catefitsurv(data = survivalExample,
                   score.method = "randomForest",
                   cate.model = Surv(y, d) ~ age + female + previous_cost +
                                             previous_number_relapses,
                   ps.model = trt ~ age + previous_treatment,
                   ipcw.model = ~ age + previous_cost + previous_treatment,
                   tau0 = tau0,
                   seed = 999)

coef(fit)
}

}
\references{
Yadlowsky, S., Pellegrini, F., Lionetto, F., Braune, S., & Tian, L. (2020).
\emph{Estimation and validation of ratio-based conditional average treatment effects using
observational data. Journal of the American Statistical Association, 1-18.} DOI: 10.1080/01621459.2020.1772080.
}
\seealso{
\code{\link{catecvsurv}()}
}
