% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pvarx.R
\name{pvarx}
\alias{pvarx}
\alias{pvarx.VAR}
\alias{pvarx.VEC}
\title{Estimation of VAR models for heterogeneous panels}
\usage{
pvarx.VAR(
  L.data,
  lags,
  type = c("const", "trend", "both", "none"),
  t_D = NULL,
  D = NULL,
  n.factors = FALSE,
  n.iterations = FALSE
)

pvarx.VEC(
  L.data,
  lags,
  dim_r,
  type = c("Case1", "Case2", "Case3", "Case4", "Case5"),
  t_D1 = NULL,
  t_D2 = NULL,
  D1 = NULL,
  D2 = NULL,
  idx_pool = NULL,
  n.factors = FALSE,
  n.iterations = FALSE
)
}
\arguments{
\item{L.data}{List of '\code{data.frame}' objects for each individual. 
The variables must have the same succession \eqn{k = 1,\ldots,K} 
in each individual '\code{data.frame}'.}

\item{lags}{Integer or vector of integers. 
Lag-order of the VAR models in levels, which is
either a common \eqn{p} for all individuals or 
individual-specific \eqn{p_i} for each individual.  
In the vector, \eqn{p_i} must have the same succession 
\eqn{i = 1,\ldots,N} as argument '\code{L.data}'.}

\item{type}{Character. The conventional case of the \link[=as.t_D]{deterministic term}.}

\item{t_D}{List of vectors. The activating break periods \eqn{\tau} 
for the period-specific \link[=as.t_D]{deterministic regressors} 
in \eqn{d_{it}} of the VAR model in levels.}

\item{D}{List. A single '\code{data.frame}' of common deterministic 
regressors or a list of \eqn{N} '\code{data.frame}' objects of the 
individual \link[=as.t_D]{deterministic regressors} added to \eqn{d_{it}}. 
These customized regressors correspond to '\code{exogen}' in \strong{vars}' 
\code{\link[vars]{VAR}}, which is fixed over bootstrap iterations.}

\item{n.factors}{Integer. Number of common factors to be used for SUR. 
Deactivated if \code{FALSE} (the default).}

\item{n.iterations}{Integer. The (maximum) number of iterations 
for the estimation of SUR resp. the pooled cointegrating vectors.}

\item{dim_r}{Integer. Common cointegration-rank \eqn{r} of the VECM.}

\item{t_D1}{List of vectors. The activating break periods \eqn{\tau} 
for the period-specific \link[=as.t_D]{deterministic regressors} 
in \eqn{d_{1,it}}, which are restricted to the cointegration relations. 
Just as in \code{\link{pcoint}}, the 
accompanying lagged regressors are automatically included in \eqn{d_{2,it}}.}

\item{t_D2}{List of vectors. The activating break periods \eqn{\tau} 
for the period-specific \link[=as.t_D]{deterministic regressors} 
in \eqn{d_{2,it}}, which are unrestricted.}

\item{D1}{List. A single '\code{data.frame}' of common deterministic regressors
regressors or a list of \eqn{N} '\code{data.frame}' objects of 
individual \link[=as.t_D]{deterministic regressors} added 
to \eqn{d_{1,it}}, which are restricted to the cointegration relations.
Unlike '\code{t_D1}', these customized regressors require potential
accompanying lagged regressors to be manually included in \eqn{d_{2,it}}}

\item{D2}{List. A single '\code{data.frame}' of common deterministic 
regressors or a list of \eqn{N} '\code{data.frame}' objects of 
individual \link[=as.t_D]{deterministic regressors} added 
to \eqn{d_{2,it}}, which are unrestricted. 
These customized regressors correspond to '\code{dumvar}' in \strong{urca}'s 
\code{\link[urca]{ca.jo}}, which is fixed over bootstrap iterations.}

\item{idx_pool}{Vector. Position \eqn{k = 1,...,K+n_1} of each variable 
to be pooled using the two-step estimator by Breitung (2005). 
The integer vector specifies throughout heterogeneous coefficients up to 
the uniform upper block \eqn{I_{r}} estimated with the individual estimator 
by Ahn and Reinsel (1990) if exclusively in the interval \eqn{[0,...,r]}. 
Deactivated if \code{NULL} (the default).}
}
\value{
A list of class '\code{\link[=as.pvarx]{pvarx}}' with the elements:
\item{A}{Matrix. The lined-up coefficient matrices \eqn{A_j, j=1,\ldots,p} 
        for the lagged variables in the panel VAR estimated by mean-group.}
\item{B}{Matrix. Placeholder for the structural impact matrix.}
\item{beta}{Matrix. The \eqn{((K+n_{d1}) \times r)} cointegrating matrix
  of the VAR model if transformed from a rank-restricted VECM.}
\item{L.varx}{List of '\code{varx}' objects for the individual estimation results.}
\item{L.data}{List of '\code{data.frame}' objects for each individual.}
\item{CSD}{List of measures for cross-sectional dependency. 
  \code{NULL} if the individual VAR models have been estimated under independence.}
\item{args_pvarx}{List of characters and integers 
  indicating the estimator and specifications that have been used.}
}
\description{
Performs the (pooled) mean-group estimation of a panel VAR model.
  First, VAR models are estimated for all \eqn{N} individual entities. 
  Then, their (pooled) mean-group estimate is calculated for each coefficient.
}
\section{Functions}{
\itemize{
\item \code{pvarx.VAR()}: Mean Group (MG) of VAR models in levels.

\item \code{pvarx.VEC()}: (Pooled) Mean Group (PMG) of VECM.

}}
\examples{
data("PCAP")
names_k = c("g", "k", "l", "y")  # variable names
names_i = levels(PCAP$id_i)      # country names 
L.data  = sapply(names_i, FUN=function(i) 
  ts(PCAP[PCAP$id_i==i, names_k], start=1960, end=2019, frequency=1), 
  simplify=FALSE)
R.lags = c(2, 4, 2, 3, 2, 4, 4, 2, 2, 3, 3, 3, 2, 4, 4, 2, 2, 2, 4, 2, 2, 2, 4)
names(R.lags) = names_i

### MG of VAR by OLS ###
R.t_D  = list(t_shift=10)  # common level shift for all countries 
R.pvar = pvarx.VAR(L.data, lags=R.lags, type="both", t_D=R.t_D)
R.pirf = irf(R.pvar, n.ahead=50)  # MG of individual forecast-error IRF
plot(R.pirf)

### Pooled MG of rank-restricted VAR ###
R.pvec = pvarx.VEC(L.data, lags=R.lags, dim_r=2, idx_pool=1:4, type="Case4")
R.pirf = irf(R.pvec, n.ahead=50)  # MG of individual forecast-error IRF
plot(R.pirf)

}
\references{
Canova, F., and Ciccarelli, M. (2013): 
  "Panel Vector Autoregressive Models: A Survey",
  \emph{Advances in Econometrics}, 32, pp. 205-246.

Hsiao, C. (2014): 
  \emph{Analysis of Panel Data}, 
  Econometric Society Monographs, Cambridge University Press, 3rd ed.

Luetkepohl, H. (2005): 
  \emph{New Introduction to Multiple Time Series Analysis}, 
  Springer, 2nd ed.

Pesaran, M. H., and Smith R. J. (1995): 
  "Estimating Long-Run Relationships from Dynamic Heterogeneous Panels",
  \emph{Journal of Econometrics}, 68, pp. 79-113.

Rebucci, A. (2010): 
  "Estimating VARs with Long Stationary Heterogeneous Panels: 
  A Comparison of the Fixed Effect and the Mean Group Estimators",
  \emph{Economic Modelling}, 27, pp. 1183-1198.

Ahn, S. K., and Reinsel (1990): 
  "Estimation for Partially Nonstationary Multivariate Autoregressive Models",
  \emph{Journal of the American Statistical Association}, 85, pp. 813-823.

Breitung, J. (2005): 
  "A Parametric Approach to the Estimation of Cointegration Vectors in Panel Data",
  \emph{Econometric Reviews}, 24, pp. 151-173.

Johansen, S. (1996): 
  \emph{Likelihood-based Inference in Cointegrated Vector Autoregressive Models}, 
  Advanced Texts in Econometrics, Oxford University Press, USA.

Pesaran, M. H., Shin, Y, and Smith R. J. (1999): 
  "Pooled Mean Group Estimation of Dynamic Heterogeneous Panels",
  \emph{Journal of the American Statistical Association}, 94, pp. 621-634.
}
\concept{panel estimation functions}
