% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pref-eval.r
\name{psel}
\alias{psel}
\alias{psel.indices}
\alias{peval}
\title{Preference Selection}
\usage{
psel(df, pref, ...)

psel.indices(df, pref, ...)

peval(pref, ...)
}
\arguments{
\item{df}{A data frame, data frame extension (e.g. a tibble), or a grouped data frame from \code{\link[dplyr]{group_by}}.}

\item{pref}{A preference object. See \code{\link{complex_pref}} and \code{\link{base_pref}} for details.
All variables occurring in the definition of \code{pref} must be either columns of \code{df}
or variables/functions of the environment where \code{pref} was defined.}

\item{...}{Additional optional parameters:
 \describe{
   \item{\code{top}}{Integer. A \code{top} value of k means that the k-best tuples of the data set are returned.
     For \code{top = Inf} all tuples are returned.}
   \item{\code{at_least}}{Integer. An \code{at_least} value of k returns the top-k tuples and additionally all tuples which are
     not dominated by the worst tuple (i.e. the minima) of the Top-k set.
     The number of tuples returned is greater or equal than
     \code{at_least}. In contrast to top-k, this is deterministic.}
   \item{\code{top_level}}{Integer. A \code{top_level} value of k returns all tuples from the k-best levels. See below for the definition of a level.}
   \item{\code{and_connected}}{Logical value. This is only relevant if more than one of the above \{\code{top}, \code{at_least}, \code{top_level}\}
     values is given, otherwise it will be ignored.
     Then \code{and_connected = TRUE} (which is the default) means that all top-conditions
     must hold for the returned tuples.}
   \item{\code{show_level}}{Logical value. If \code{TRUE}, a column \code{.level}
     is added to the result, containing all level values.
     If at least one of the \{\code{top}, \code{at_least}, \code{top_level}\} values are given,
     then \code{show_level} is \code{TRUE} by default for the \code{psel} function.
     Otherwise, and for \code{psel.indices} in all cases, this option is \code{FALSE} by default.}
   \item{\code{show_index}}{Logical value. If \code{TRUE}, a column \code{.index} is added to the result.
     Not applicable for \code{psel.indices}.}
}}
}
\description{
Evaluates a preference on a given data set, i.e.,
returns the maximal elements of a data set for a given preference order.
}
\details{
The difference between the three variants of the preference selection is:

\itemize{
  \item{\code{psel}: Returns a subset of the data set containing the maxima according to the given preference.}
  \item{\code{psel.indices}: Returns just the row indices of the maxima (except top-k queries with \code{show_level = TRUE}, see top-k preference selection).
        Hence, \code{psel(df, pref)} is equivalent to \code{df[psel.indices(df, pref), ]} for non-grouped data frames.}
  \item{\code{peval}: Does the same as \code{psel}, but assumes that \code{pref} has an associated data frame which is used for the preference selection.
        See \code{\link{base_pref}} for details, or use \code{\link{assoc.df}} to explicitly associate a preference with a data frame.}
}
}
\section{Top-k Preference Selection}{


For a given \code{top} value of k the k best elements and their level values (column \code{.level}) are returned.
The elements of level k are also called the n-th stratum in the literature.
The level values are determined as follows:

\itemize{
\item{All the maxima of a data set w.r.t. a preference have level 1.}
\item{The maxima of the remainder, i.e., the data set without the level 1 maxima, have level 2.}
\item{The n-th iteration of "Take the maxima from the remainder" returns tuples of level n.}
}

By default, \code{psel.indices} does not return the level values. By setting \code{show_level = TRUE} this function
returns a data frame with the columns '.index' and '.level'.
Note that, if none of the top-k values \{\code{top}, \code{at_least}, \code{top_level}\} is set,
then all level values are equal to 1.

By definition, a top-k preference selection is non-deterministic.
A top-1 query of two equivalent tuples (equivalence according to \code{pref})
can return both of these tuples.

Consider the following example:

\code{df <- data.frame(a = c(1, 1, 3), b = c(1, 2, 3))}

The query \code{psel(df, low(a))} returns:

\preformatted{
    a b
  1 1 1
  2 1 2
}

The top-1 \code{psel(df, low(a), top = 1)} selection returns:

\preformatted{
    a b .level
  1 1 1      1
}

Theoretically, the b=2 could also be returned in the above query.
On the contrary, a preference selection using \code{at_least} is deterministic by adding all tuples having the same level as the worst level
of the corresponding top-k query. This means, the result is filled with all tuples being not worse than the top-k result.
A preference selection with top-level-k returns all tuples having level k or better.

If the \code{top} or \code{at_least} value is greater than the number of elements in \code{df}
(i.e., \code{nrow(df)}), or \code{top_level} is greater than the highest level in \code{df},
then all elements of \code{df} will be returned without further warning.
In addition, the we can set \code{top = Inf} to return all tuples, i.e., \code{psel(df, low(a), top = Inf)} returns:

\preformatted{
    a b .level
  1 1 1      1
  2 1 2      1
  3 3 3      2
}

By setting \code{top_level = 2} we return the first two levels, i.e., all tuples in this case.

If multiple top-k parameters are specified, their interaction is controlled by \code{and_connected}.
Let \code{cond1} and \code{cond2} be top-conditions like \code{top=2} or \code{top_level=3}, then
\code{psel([...], cond1, cond2)} is equivalent to the intersection of \code{psel([...], cond1)} and \code{psel([...], cond2)}. If we have
\code{and_connected = FALSE}, these conditions are or-connected.
This corresponds to the union of \code{psel([...], cond1)} and \code{psel([...], cond2)}.
}

\section{Grouped Preference Selection}{


Using \code{psel} it is also possible to perform a preference selection where the maxima are calculated for every group separately.
The groups have to be created with \code{\link[dplyr]{group_by}} from the dplyr package. The preference selection preserves the grouping, i.e.,
the groups are restored after the preference selection.

For example, if the \code{summarize} function from dplyr is applied to
\code{psel(group_by(...), pref)}, the summarizing is done for the set of maxima of each group.
This can be used to e.g., calculate the number of maxima in each group, see the examples below.

A \{\code{top}, \code{at_least}, \code{top_level}\} preference selection
is applied to each group separately.
A \code{top=k} selection returns the k best tuples for each group.
Hence if there are 3 groups in \code{df}, each containing at least 2 elements,
and we have \code{top = 2}, then 6 tuples will be returned.
}

\section{Parallel Computation}{


On multi-core machines the preference selection can be run in parallel using a divide-and-conquer approach.
Depending on the data set, this may be faster than a single-threaded computation.
To activate parallel computation within rPref the following option has to be set:

\code{options(rPref.parallel = TRUE)}

If this option is not set, rPref will use single-threaded computation by default.
With the option \code{rPref.parallel.threads} the maximum number of threads can be specified.
The default is the number of cores on your machine.
To set the number of threads to the value of 4, use:

\code{options(rPref.parallel.threads = 4)}
}

\examples{

# Skyline and top-k/at-least Skyline
psel(mtcars, low(mpg) * low(hp))
psel(mtcars, low(mpg) * low(hp), top = 5)
psel(mtcars, low(mpg) * low(hp), at_least = 5, show_index = TRUE)

# Preference with associated data frame and evaluation
p <- low(mpg, df = mtcars) * (high(cyl) & high(gear))
peval(p)

# Visualizes the Skyline in a plot.
sky1 <- psel(mtcars, high(mpg) * high(hp))
plot(mtcars$mpg, mtcars$hp)
points(sky1$mpg, sky1$hp, lwd = 3)

# Grouped preference with dplyr.
library(dplyr)
psel(group_by(mtcars, cyl), low(mpg))

# Returns the size of each maxima group.
summarise(psel(group_by(mtcars, cyl), low(mpg)), n())

}
\seealso{
See \code{\link{complex_pref}} on how to construct a Skyline preference.
}
