% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BvM.R
\name{bvm}
\alias{bvm}
\alias{d_bvm}
\alias{const_bvm}
\alias{r_bvm}
\alias{fit_bvm_mm}
\alias{fit_bvm_mle}
\title{Density evaluation, sampling, and parameter estimation of the
bivariate sine von Mises distribution}
\usage{
d_bvm(x, mu, kappa, log_const = NULL)

const_bvm(kappa, M = 25, MC = 10000)

r_bvm(n, mu, kappa)

fit_bvm_mm(
  x,
  lower = c(0, 0, -30),
  upper = c(30, 30, 30),
  start = NULL,
  M = 25,
  hom = FALSE,
  indep = FALSE,
  ...
)

fit_bvm_mle(
  x,
  start = NULL,
  M = 25,
  lower = c(-pi, -pi, 0, 0, -30),
  upper = c(pi, pi, 30, 30, 30),
  hom = FALSE,
  indep = FALSE,
  ...
)
}
\arguments{
\item{x}{matrix of size \code{c(nx, 2)} with the angles on which the density
is evaluated.}

\item{mu}{circular means of the density, a vector of length \code{2}.}

\item{kappa}{vector of length \code{3} with the concentrations
\eqn{(\kappa_1, \kappa_2)} and the dependence parameter \eqn{\lambda}
of the density.}

\item{log_const}{logarithm of the normalizing constant. Computed internally
if \code{NULL} (default).}

\item{M}{truncation of the series expansion for computing the normalizing
constant. Defaults to \code{25}.}

\item{MC}{Monte Carlo replicates for computing the normalizing
constant when there is no series expansion. Defaults to \code{1e4}.}

\item{n}{sample size.}

\item{lower, upper}{vectors of length \code{5} with the bounds for the
likelihood optimizer. Default to \code{c(-pi, -pi, 0, 0, -30)} and
\code{c(pi, pi, 30, 30, 30)}.}

\item{start}{a vector of length \code{5} with the initial values for the
maximum likelihood optimizer. The first two entries are disregarded in
\code{fit_bvm_mm}. If \code{NULL} (default), the starting values are taken
from the estimation of marginal von Mises in \code{fit_bvm_mm}. In
\code{fit_bvm_mle}, the method of moments estimates are employed.}

\item{hom}{assume a homogeneous distribution with equal marginal
concentrations? Defaults to \code{FALSE}.}

\item{indep}{set the dependence parameter to zero? Defaults to \code{FALSE}.}

\item{...}{further parameters passed to
\code{\link[sdetorus]{mleOptimWrapper}}.}
}
\value{
\itemize{
  \item \code{d_bvm}: a vector of length \code{nx} with the density evaluated
  at \code{x}.
  \item \code{const_bvm}: the value of the normalizing constant
  \eqn{T(\kappa_1, \kappa_2, \lambda)}.
  \item \code{r_bvm}: a matrix of size \code{c(n, 2)} with the random sample.
  \item \code{fit_mme_bvm, fit_mle_bvm}: a list with the estimated parameters
  \eqn{(\mu_1, \mu_2, \kappa_1, \kappa_2, \lambda)} and the object \code{opt}
  containing the optimization summary.
}
}
\description{
Computation of the density and normalizing constant
\eqn{T(\kappa_1, \kappa_2, \lambda)} of the bivariate sine von Mises
\deqn{f(\theta_1, \theta_2)= T(\kappa_1, \kappa_2, \lambda)
\exp\{\kappa_1 \cos(\theta_1-\mu_1) +
\kappa_2 \cos(\theta_2-\mu_2) +
\lambda \sin(\theta_1-\mu_1) \sin(\theta_2-\mu_2)\}.}

Simulation of samples from a bivariate sine von Mises.

Maximum likelihood and method of moments estimation of the
parameters \eqn{(\mu_1, \mu_2, \kappa_1, \kappa_2, \lambda)}.
}
\examples{
## Density evaluation

mu <- c(0, 0)
kappa <- 3:1
nth <- 50
th <- seq(-pi, pi, l = nth)
x <- as.matrix(expand.grid(th, th))
const <- const_bvm(kappa = kappa)
d <- d_bvm(x = x, mu = mu, kappa = kappa, log_const = log(const))
filled.contour(th, th, matrix(d, nth, nth), col = viridisLite::viridis(31),
               levels = seq(0, max(d), l = 30))

## Sampling and estimation

n <- 100
samp <- r_bvm(n = n, mu = mu, kappa = kappa)
(param_mm <- fit_bvm_mm(samp)$par)
(param_mle <- fit_bvm_mle(samp)$par)
}
\references{
Mardia, K. V., Hughes, G., Taylor, C. C., and Singh, H. (2008).
A multivariate von Mises with applications to bioinformatics.
\emph{Canadian Journal of Statistics}, 36(1):99--109.
\doi{10.1002/cjs.5550360110}

Singh, H., Hnizdo, V., and Demchuk, E. (2002). Probabilistic model for two
dependent circular variables. \emph{Biometrika}, 89(3):719--723.
\doi{10.1093/biomet/89.3.719}
}
