% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ridge-pca.R
\name{ridge_pca}
\alias{ridge_pca}
\title{Toroidal PCA via density ridges}
\usage{
ridge_pca(
  x,
  type = c("auto", "bvm", "bwc")[1],
  N = 500,
  K = 15,
  scale = TRUE,
  lrts = TRUE,
  alpha = 0.05,
  at2 = TRUE,
  ...
)
}
\arguments{
\item{x}{matrix of dimension  \code{c(n, 2)} containing the \code{n}
observations of the pair of angles.}

\item{type}{either \code{"bvm"} (bivariate sine von Mises), \code{"bwc"}
(bivariate wrapped Cauchy), or \code{"auto"} (default). \code{"auto"}
performs both fits and uses the one with lowest BIC.}

\item{N}{number of discretization points for approximating curve lengths.
Defaults to \code{5e2}.}

\item{K}{number of terms in the Fourier expansion. Defaults to \code{15}.}

\item{scale}{scale the resulting scores to \eqn{[-\pi, \pi)^2}? Defaults
to \code{TRUE}.}

\item{lrts}{run \code{\link{biv_lrt}} to check the null hypothesis of
homogeneous concentration parameters using likelihood ratio tests? If
\code{TRUE} (default), enforces the special horizontal/vertical/diagonal
cases to become "sticky" fits.}

\item{alpha}{significance level for the homogeneity test.}

\item{at2}{do the \code{atan2} fit instead of the sine fit (only using
\eqn{S_m})? Defaults to \code{TRUE}. \code{at2 = FALSE} is not
recommended to use.}

\item{...}{optional parameters passed to \code{\link{fit_bvm_mle}} and
\code{\link{fit_bwc_mle}}, such as \code{start}, \code{lower}, or
\code{upper}.}
}
\value{
A list with:
\item{mu_hat}{estimated circular means of the sample.}
\item{coefs_hat}{estimated Fourier coefficients.}
\item{ind_var}{indexing variable.}
\item{scores}{scores for each of the sample points.}
\item{var_exp}{percentage of explained variance.}
\item{fit_mle}{maximum likelihood fit.}
\item{bic_fit}{BIC of the fit.}
\item{data}{original sample.}
\item{scales}{vector of length 2 with the scale limits for the axes.}
\item{type}{type of fit performed.}
\item{p_hom}{\eqn{p}-value of the homogeneity test.}
\item{p_indep}{\eqn{p}-value of the independence test.}
}
\description{
This function computes the whole process of toroidal PCA
via density ridges on a given sample: parameter estimation of the
underlying distribution, estimation of the connected component of the ridge,
and determination of its Fourier expansion from which to obtain the first
and second scores.
}
\examples{
\donttest{
## Bivariate von Mises

n <- 100
x <- r_bvm(n = n, mu = c(1, 2), kappa = c(0.4, 0.4, 0.5))
fit <- ridge_pca(x = x, type = "bvm")
show_ridge_pca(fit = fit, col_data = "red")

x <- r_bvm(n = n, mu = c(2, 1), kappa = c(1, 2, 0))
fit <- ridge_pca(x = x, type = "bvm")
show_ridge_pca(fit = fit, col_data = "red")

x <- r_bvm(n = n, mu = c(2, 1), kappa = c(3, 2, 0))
fit <- ridge_pca(x = x, type = "bvm")
show_ridge_pca(fit = fit, col_data = "red")

## Bivariate wrapped Cauchy

x <- r_bwc(n = n, mu = c(1, 2), xi = c(0.2, 0.2, 0.5))
fit <- ridge_pca(x = x, type = "bwc")
show_ridge_pca(fit = fit, col_data = "red")

x <- r_bwc(n = n, mu = c(1, 2), xi = c(0.2, 0.8, 0))
fit <- ridge_pca(x = x, type = "bwc")
show_ridge_pca(fit = fit, col_data = "red")

x <- r_bwc(n = n, mu = c(1, 2), xi = c(0.5, 0.2, 0))
fit <- ridge_pca(x = x, type = "bwc")
show_ridge_pca(fit = fit, col_data = "red")}
}
