% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/session.R
\name{selenider_session}
\alias{selenider_session}
\title{Start a session}
\usage{
selenider_session(
  session = getOption("selenider.session"),
  browser = getOption("selenider.browser"),
  timeout = 4,
  options = NULL,
  driver = NULL,
  local = TRUE,
  .env = rlang::caller_env(),
  view = FALSE,
  selenium_manager = TRUE,
  quiet = TRUE
)
}
\arguments{
\item{session}{The package to use as a backend: either "chromote",
"selenium". By default, chromote is used, since this tends to be faster
and more reliable. Change the default value using the \code{selenider.session}
option.}

\item{browser}{The name of the browser to run the session in; one of
"chrome", "firefox", "edge", "safari", or another valid browser name.
If \code{NULL}, the function will try to work out which browser you have
installed. If we are using chromote, this option is ignored, since
chromote only works on Chrome. Change the default value of this parameter
using the \code{selenider.browser} option.}

\item{timeout}{The default time to wait when collecting an element.}

\item{options}{A \code{\link[=chromote_options]{chromote_options()}} or \code{\link[=selenium_options]{selenium_options()}} object,
used to specify options that are specific to chromote or selenium. See
Details for some useful examples of this.}

\item{driver}{A driver object to use instead of creating one manually. This
can be one of:
\itemize{
\item A \link[chromote:ChromoteSession]{chromote::ChromoteSession} object.
\item A \link[shinytest2:AppDriver]{shinytest2::AppDriver} object.
\item An \link[selenium:SeleniumSession]{selenium::SeleniumSession} object.
\item A Selenium server object, created by \code{\link[selenium:selenium_server]{selenium::selenium_server()}}. In
this case, a client will be created using the server object.
\item A list/environment containing the \link[selenium:SeleniumSession]{selenium::SeleniumSession} object,
the Selenium server object, or both.
}}

\item{local}{Whether to set the session as the local session object,
using \code{\link[=local_session]{local_session()}}. If this is \code{FALSE}, you will have to pass this
into the \code{session} argument of other functions (like \code{\link[=open_url]{open_url()}}), and
close the session manually using \code{\link[=close_session]{close_session()}}.}

\item{.env}{Passed into \code{\link[=local_session]{local_session()}}, to define the
environment in which the session is used. Change this if you want to
create the session inside a function and then use it outside the
function (see Examples). Use \code{\link[rlang:stack]{rlang::caller_env()}} in this case.}

\item{view, selenium_manager, quiet}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}}
Use the \code{options} argument instead.}
}
\value{
A \code{selenider_session} object. Use \code{session$driver} to retrieve the driver
object that controls the browser.
}
\description{
Create a session in selenider. If you create a session in the global
environment it will be made available to other functions (like
\code{\link[=open_url]{open_url()}}) without having to pass it in, and will close automatically
when the R session is closed. Alternatively, if it is created inside a
function, it will be closed as soon as the function finishes executing. To
customise this, use the \code{.env} and \code{local} arguments.
}
\section{Useful session-specific options}{
See \code{\link[=chromote_options]{chromote_options()}} and \code{\link[=selenium_options]{selenium_options()}} for the full range.
\subsection{Making a chromote session non-headless}{

By default, chromote will run in headless mode, meaning that you won't
actually be able to see the browser as you control it. Use the \code{view}
argument to \code{\link[=chromote_options]{chromote_options()}} to change this:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{session <- selenider_session(
  options = chromote_options(view = TRUE)
)
}\if{html}{\out{</div>}}
}

\subsection{Prevent creation of a selenium server}{

Sometimes, you want to manage the Selenium server separately, and only let
selenider create client objects to attach to the server. You can do this by
passing \code{NULL} into the \code{server_options} argument to \code{\link[=selenium_options]{selenium_options()}}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{session <- selenider_session(
  "selenium",
  options = selenium_options(server_options = NULL)
)
}\if{html}{\out{</div>}}

If the port you are using is not 4444, you will need to pass in the \code{port}
argument to \code{\link[=selenium_client_options]{selenium_client_options()}} as well:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{session <- selenider_session(
  "selenium",
  options = selenium_options(
    client_options = selenium_client_options(port = YOUR_PORT),
    server_options = NULL
  )
)
}\if{html}{\out{</div>}}

One example of when this may be useful is when you are managing the Selenium
server using Docker.
}

\subsection{Store the Selenium server persistently}{

By default, selenium will download and store the Selenium server JAR file
in a temporary directory, which will be deleted when the R session finishes.
This means that every time you start a new R session, this file will be
re-downloaded. You can store the JAR file permanently using the \code{temp}
argument to \code{\link[=selenium_server_options]{selenium_server_options()}}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{session <- selenider_session(
  "selenium",
  options = selenium_options(
    server_options = selenium_server_options(temp = TRUE)
  )
)
}\if{html}{\out{</div>}}

The downside of this is you may end up using a lot of storage, especially
if a new version of Selenium is released and the old server file is left
on the filesystem.

You can also use the \code{path} argument to \code{\link[=selenium_server_options]{selenium_server_options()}} to
specify the directory where the JAR file should be stored.
}
}

\section{Structure of a selenider session}{
A \code{selenider_session} object has several components that can be useful to
access:
\itemize{
\item \code{session} - The type of session, either \code{"chromote"} or \code{"selenium"}.
\item \code{driver} - The driver object used to control the browser. This is either a
\link[chromote:ChromoteSession]{chromote::ChromoteSession} or \link[selenium:SeleniumSession]{selenium::SeleniumSession} object. This is
useful if you want to do something with the driver that is not directly
supported by selenider. See \code{\link[=get_actual_element]{get_actual_element()}} for some examples of
this.
\item \code{server} - The Selenium server object, if one was created or passed in.
\item \code{id} - A unique ID that can be used to identify the session.
}

Access these components using \code{$} (e.g. \code{session$driver}).
}

\section{Custom drivers}{
If you want complete manual control over creating the underlying driver,
you can pass your own \code{driver} argument to stop selenider from creating
the driver for you.

You can also supply a \link[shinytest2:AppDriver]{shinytest2::AppDriver} object, allowing selenider and
shinytest2 to share a session:

\if{html}{\out{<div class="sourceCode">}}\preformatted{shiny_app <- shiny::shinyApp(
  ui = shiny::fluidPage(
    # ... Your UI
  ),
  server = function(input, output) \{
    # ... Your server
  \}
)

app <- shinytest2::AppDriver$new()

session <- selenider_session(
  driver = app
)
}\if{html}{\out{</div>}}
}

\examples{
\dontshow{if (selenider::selenider_available()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

session_1 <- selenider_session(timeout = 10)
# session_1 is the local session here

get_session() # Returns session 1

my_function <- function() {
  session_2 <- selenider_session()

  # In here, session_2 is the local session
  get_session()
} # When the function finishes executing, the session is closed

my_function() # Returns `session_2`

# If we want to use a session outside the scope of a function,
# we need to use the `.env` argument.
create_session <- function(timeout = 10, .env = rlang::caller_env()) {
  # caller_env() is the environment where the function is called
  selenider_session(timeout = timeout, .env = .env)
}

my_session <- create_session()

# We can now use this session outside the `create_session()` function
get_session()

# `my_session` will be closed automatically.
\dontshow{\}) # examplesIf}
}
\seealso{
\itemize{
\item \code{\link[=close_session]{close_session()}} to close the session. Note that this will not reset the
result of \code{\link[=get_session]{get_session()}}, which is why \code{\link[withr:defer]{withr::deferred_run()}} is
preferred.
\item \code{\link[=local_session]{local_session()}} and \code{\link[=with_session]{with_session()}} to manually set the local session
object (and \code{\link[=get_session]{get_session()}} to get it).
\item \code{\link[=open_url]{open_url()}}, \code{\link[=s]{s()}} and \code{\link[=find_elements]{find_elements()}} to get started once you have
created a session.
}
}
