% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/allocative.efficiency.R
\name{allocative.efficiency}
\alias{allocative.efficiency}
\title{Allocative efficiency estimation}
\usage{
allocative.efficiency(X, y, X.price, y.price, X.constrained = NA,
  H.inv = NA, H.mult = 1, model = "br", method = "u",
  scale.constraints = TRUE)
}
\arguments{
\item{X}{Matrix of inputs}

\item{y}{Vector of outputs}

\item{X.price}{Matrix of input prices}

\item{y.price}{Vector of output prices}

\item{X.constrained}{Matrix of inputs where constraints apply}

\item{H.inv}{Inverse of the smoothing matrix (must be positive definite); defaults to rule of thumb}

\item{H.mult}{Scaling factor for rule of thumb smoothing matrix}

\item{model}{Type of frontier to use; "br" for boundary regression, "sf" for stochastic frontier}

\item{method}{Constraints to apply; "u" for unconstrained, "m" for monotonically increasing, and "mc" for monotonically increasing and concave}

\item{scale.constraints}{Boolean, whether to scale constraints by their average value, can help with convergence}
}
\value{
Returns a list with the following elements
\item{y.fit}{Estimated value of the frontier at X.fit}
\item{gradient.fit}{Estimated gradient of the frontier at X.fit}
\item{technical.efficiency}{Estimated technical efficiency}
\item{log.overallocation}{Estimated log-overallocation of each input for each observation}
\item{X.eval}{Matrix of inputs used for fitting}
\item{X.constrained}{Matrix of inputs where constraints apply}
\item{H.inv}{Inverse smoothing matrix used in fitting}
\item{method}{Method used to fit frontier}
\item{scaling.factor}{Factor by which constraints are multiplied before quadratic programming}
}
\description{
Fits frontier to data and estimates technical and allocative efficiency
}
\details{
This function estimates allocative inefficiency using the methodology in McKenzie
(2018). The estimation process is a non-parametric analogue of Schmidt and Lovell
(1979). First, the frontier is fit using either a boundary regression or stochastic
frontier as in Racine et al. (2009), from which technical efficiency is estimated.
Then, gradients and price ratios are computed for each observation and compared to 
determine the extent of misallocation. Specifically, log-overallocation is computed as

\deqn{\log\left(\frac{w_i^j}{p_i}\right) - \log\left(\phi_i\frac{\partial f(x_i)}{\partial x^j}\right),}

where \eqn{\phi_i} is the efficiency of observation \eqn{i},
\eqn{\partial f(x_i) / \partial x^j} is the marginal productivity of input \eqn{j}
at observation \eqn{i}, \eqn{w_i^j} is the cost of input \eqn{j} for observation
\eqn{i}, and \eqn{p_i} is the price of output for observation \eqn{i}.
}
\examples{
data(USMacro)

USMacro <- USMacro[complete.cases(USMacro),]

#Extract data
X <- as.matrix(USMacro[,c("K", "L")])
y <- USMacro$Y

X.price <- as.matrix(USMacro[,c("K.price", "L.price")])
y.price <- rep(1e9, nrow(USMacro)) #Price of $1 billion of output is $1 billion

#Run model
efficiency.model <- allocative.efficiency(X, y,
                                         X.price, y.price,
                                         X.constrained = X,
                                         model = "br",
                                         method = "mc")

#Plot technical/allocative efficiency over time
library(ggplot2)

technical.df <- data.frame(Year = USMacro$Year,
                          Efficiency = efficiency.model$technical.efficiency)

ggplot(technical.df, aes(Year, Efficiency)) +
  geom_line()

allocative.df <- data.frame(Year = rep(USMacro$Year, times = 2),
                            log.overallocation = c(efficiency.model$log.overallocation[,1],
                                                   efficiency.model$log.overallocation[,2]),
                            Variable = rep(c("K", "L"), each = nrow(USMacro)))

ggplot(allocative.df, aes(Year, log.overallocation)) +
  geom_line(aes(color = Variable))

#Estimate average overallocation across sample period
lm.model <- lm(log.overallocation ~ 0 + Variable, allocative.df)
summary(lm.model)
  
}
\references{
\insertRef{AignerLovellSchmidt}{snfa}\cr\cr
\insertRef{McKenzie}{snfa}\cr\cr
\insertRef{ParmeterRacine}{snfa}\cr\cr
\insertRef{SchmidtLovell}{snfa}
}
