% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_agecomp.R
\name{sample_agecomp}
\alias{sample_agecomp}
\title{Sample age compositions from a Stock Synthesis data file}
\usage{
sample_agecomp(dat_list, outfile = NULL, fleets, Nsamp, years,
  cpar = 1, ESS = NULL, keep_conditional = TRUE)
}
\arguments{
\item{dat_list}{An SS data list object as read in from
\code{\link[r4ss]{SS_readdat}} in the \pkg{r4ss} package. Make sure
you select option \code{section=2}.}

\item{outfile}{A character string specifying the file name to use
when writing the information to the disk. The string must include
the proper file extension. No file is written using the default value
of \code{NULL}, which leads to increased speed because writing the 
file takes time and computing resources.}

\item{fleets}{*A vector of integers specifying which fleets to include.
The order of the fleets pertains to the input order of other arguments.
An entry of \code{fleets=NULL} leads to zero samples for any fleet.}

\item{Nsamp}{*A numeric list of the same length as \code{fleets}.
Either single values or vectors of the same length as the number of
years can be passed through. Single values are repeated for all
years. If no fleet collected samples, keep the value to
\code{Nsamp=NULL}.}

\item{years}{*A list the same length as \code{fleets} giving the years as
numeric vectors. If no fleet collected samples, keep the value to
\code{years=NULL}.}

\item{cpar}{A numeric value or vector the same length as
\code{fleets} controlling the variance of the Dirichlet
distribution used for sampling. A value of \code{1} leads to the
same standard deviation as a multinomial of the given \code{Nsamp},
\code{2} indicates twice, etc. Values greater than one indicate
overdispersion, and less underdispersion. \code{NULL} or \code{NA}
for a given fleet will lead to no dispersion.}

\item{ESS}{The final effective sample size (ESS) associated with the
simulated data. The ESS is not used to generate the simulated data
but can be used as an input sample size in subsequent models that estimate
population parameters or status.
The default, \code{NULL}, leads to the true (internally calculated)
#' ESS being used, which is \code{Nsamp} for the multinomial case or given
by the formula under \code{cpar} for the Dirichlet case.
At least one value must be provided for each fleet or a vector of
year-specific values can be used for any given fleet.
The argument accepts a list with entries,
either a single integer or a vector of integers, for each fleet.}

\item{keep_conditional}{A logical if conditional age-at-length data
should be kept or removed entirely from the data file.
\code{sample_agecomp} only works on the age-composition data
and not on the conditional age-at-length data. To sample the
conditional data, set \code{keep_conditional} to \code{TRUE}
and use \code{\link{sample_calcomp}}.}
}
\value{
A modified \code{.dat} file if \code{!is.null(outfile)}. A list object
containing the modified \code{.dat} file is returned invisibly.
}
\description{
Extract age-composition data from a \code{.ss_new} data file and sample
the data. It is assumed that the composition data will be expected values
as written by Stock Synthesis in the second section of the data file, but
one can also sample input data. The resulting age-composition
data are assumed to represent observed age composition and will overwrite
the age data in \code{dat_list}, which is returned invisibly.
The data file can also be written to the disk, if a file path is provided to
\code{outfile}, and used as simulated data by an estimation model.
If used with \code{\link{run_ss3sim}}, the case file should be named
\code{agecomp}. A suggested (default) case letter is \code{D} for data.
}
\section{Which arguments to specifiy in case files}{

All function argument descriptions that start with an asterisk (*) will be passed
through the case files to \code{\link{run_ss3sim}}. If one of these arguments
is not specified in a case file, then a value of \code{NULL} will be passed,
which may or may not be an appropriate value. Other arguments will be ignored
if specified.
}

\examples{
d <- system.file("extdata", package = "ss3sim")
f_in <- file.path(d, "models", "cod-om", "codOM.dat")
dat_list <- r4ss::SS_readdat(f_in, version = NULL, verbose = FALSE)

## Turn off age comps by specifying fleets=NULL
test <- sample_agecomp(dat_list = dat_list, fleets = NULL)

## Generate with a smaller number of fleet taking samples
ex1 <- sample_agecomp(dat_list = dat_list, outfile = NULL,
  fleets = 2, Nsamp = list(c(10, 50)), years = list(c(26, 27)))
NROW(ex1$agecomp) == 2

## Generate with varying Nsamp by year for first fleet
ex2 <- sample_agecomp(dat_list = dat_list, outfile = NULL,
  fleets = c(1, 2),
  Nsamp = list(c(rep(50, 5), rep(100, 5)), 50),
  years = list(seq(26, 44, 2), c(26:100)))

## Run three  cases showing Multinomial, Dirichlet(1), and over-dispersed
## Dirichlet for different levels of sample sizes
op <- par(mfrow = c(1, 3))
set.seed(1)
true <- prop.table(dat_list$agecomp[
  dat_list$agecomp$FltSvy == 1 & dat_list$agecomp$Yr == 50, -(1:9)])
cpars <- c(NA, 1, 4)
for (samplesize in c(30, 100, 1000)) {
  if (samplesize > 30) par(mar = c(5.1, 1, 4.1, 2.1))
  plot(dat_list$agebin_vector, true, type = "b", ylim = c(0, 1),
    col = 4, lwd = 2, xlab = "Age",
    ylab = ifelse(samplesize == 30, "Proportion", ""),
    main = paste("Sample size =", samplesize))
  if (samplesize == 30) {
    legend("topright", lty = 1, col = 1:4, bty = "n",
      legend = c("Multinomial", "Dirichlet(1)", "Dirichlet(4)", "Truth"))
  }
  for (i in seq_along(cpars)) {
    ex <- sample_agecomp(dat_list = dat_list, outfile = NULL, fleets = 1,
      Nsamp = list(samplesize), years = list(50), cpar = cpars[i])$agecomp
    lines(dat_list$agebin_vector, prop.table(ex[1, -(1:9)]),
      col = i, type = "b")
  }
}
par(op)
}
\seealso{
Other sampling functions: \code{\link{clean_data}},
  \code{\link{sample_calcomp}}, \code{\link{sample_index}},
  \code{\link{sample_lcomp}}, \code{\link{sample_mlacomp}},
  \code{\link{sample_wtatage}}
}
\author{
Cole Monnahan and Kotaro Ono
}
\concept{sampling functions}
