% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/templates.R
\name{parse_logs}
\alias{parse_logs}
\alias{parse_logs_file}
\title{Parse Log Files}
\usage{
parse_logs(text, template, classes = list(), ...)

parse_logs_file(text_file, config_file, formatters = list(), ...)
}
\arguments{
\item{text}{Character vector; each element a log record}

\item{template}{Template string}

\item{classes}{A named list of parsers or regex strings for use within the
template string}

\item{...}{Other arguments passed onto \code{regexpr} for matching regular expressions.}

\item{text_file}{Filename (or readable connection) containing log text}

\item{config_file}{Filename (or readable connection) containing template file}

\item{formatters}{Named list of formatter functions for use of formatting \code{classes}}
}
\value{
A data.frame with each field identified in the template string as a column.
        For each record in the passed text, the fields were extracted and formatted
        using the parser objects in \code{default_classes()} and \code{classes}.
}
\description{
Parse a log file with a provided template and a set of classes
}
\details{
`\code{template} should only be a template string, such as
 '{{ip ip_address}} [{{date access_date}}]...'.

\code{config_file} should be a yaml file or connection with the following fields
 \itemize{
     \item template: Template String
     \item classes: Named list of regex strings for building classes
 }

\code{text} should be a character vector, with each element representing a
             a log record

\code{text_file} should be a file or connection that can be split (with readLines)
                 into a character vector of records

\code{classes} should be a named list of parser objects, where names
                match names of classes in template string, or a similarly
                named list of regex strings for coercing into parsers

\code{formatters} should be a named list of functions, where names
                match names of classes in template string, for properly
                formatting fields once they have been captured
}
\examples{
# Template string with two fields
template <- '{{ip ipAddress}} - [{{date accessDate}}] {{int status }}'

# Two simple log records
logs <- c(
  '192.168.1.10 - [26/Jul/2019:11:41:10 -0500] 200',
  '192.168.1.11 - [26/Jul/2019:11:41:21 -0500] 404'
)

# A formatter for the date field
myFormatters <- list(date = function(x) lubridate::as_datetime(x, format = '\%d/\%b/\%Y:\%H:\%M:\%S \%z'))
# A parser class for the date field
date_parser <- parser(
  '[0-3][0-9]\\\\/[A-Z][a-z]{2}\\\\/[0-9]{4}:[0-9]{2}:[0-9]{2}:[0-9]{2}[ ][\\\\+|\\\\-][0-9]{4}',
  myFormatters$date,
  'date'
)

# Parse the logs from raw data
parse_logs(logs, template, list(date=date_parser))

# Write the logs and to file and parse
logfile <- tempfile()
templatefile <- tempfile()
writeLines(logs, logfile)
yaml::write_yaml(list(template=template, classes=list(date=date_parser)), templatefile)
parse_logs_file(logfile, templatefile, myFormatters)
file.remove(logfile)
file.remove(templatefile)

}
