% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bargaining_tariff.R
\name{bargaining_tariff}
\alias{bargaining_tariff}
\title{Tariff Simulation With A Nash Bargaining Game}
\usage{
bargaining_tariff(
  demand = c("logit"),
  prices,
  shares,
  margins,
  owner = NULL,
  mktElast = NA_real_,
  insideSize = NA_real_,
  diversions,
  tariffPre = rep(0, length(shares)),
  tariffPost = rep(0, length(shares)),
  bargpowerPre = rep(0.5, length(prices)),
  bargpowerPost = bargpowerPre,
  normIndex = ifelse(isTRUE(all.equal(sum(shares), 1, check.names = FALSE)), 1, NA),
  priceOutside = ifelse(demand == "logit", 0, 1),
  priceStart,
  control.slopes,
  control.equ,
  labels = paste("Prod", 1:length(shares), sep = ""),
  ...
)
}
\arguments{
\item{demand}{A character vector indicating which demand system to use. Currently allows logit (default).}

\item{prices}{A length k vector product prices.}

\item{shares}{A length k vector of product shares. Values must be between 0 and 1.}

\item{margins}{A length k vector of product margins. All margins must be in \strong{levels} (not w.r.t to price), or NA.}

\item{owner}{EITHER a vector of length k whose values indicate which firm produced a product before the tariff OR a k x k matrix of pre-merger ownership shares.}

\item{mktElast}{A negative number equal to the industry pre-merger price elasticity. Default is NA .}

\item{insideSize}{An integer equal to total pre-merger units sold.
If shares sum to one, this also equals the size of the market.}

\item{diversions}{A k x k matrix of diversion ratios with diagonal elements equal to -1. Default is missing, in which case diversion according to revenue share is assumed.}

\item{tariffPre}{A vector of length k where each element equals the \strong{current} \emph{ad valorem} tariff
(expressed as a proportion of the consumer price) imposed on each product. Default is 0, which assumes no tariff.}

\item{tariffPost}{A vector of length k where each element equals the \strong{new}  \emph{ad valorem} tariff
(expressed as a proportion of the consumer price) imposed on each product. Default is 0, which assumes no tariff.}

\item{bargpowerPre}{A length k vector of pre-tariff bargaining power parameters. Values
must be between 0 (sellers have the power) and 1 (buyers the power). NA values are allowed,
though must be calibrated from additional margin and share data. Default is 0.5.}

\item{bargpowerPost}{A length k vector of post-tariff bargaining power parameters. Values
must be between 0 (sellers have the power) and 1 (buyers the power). NA values are allowed,
though must be calibrated from additional margin and share data. Default is \sQuote{bargpowerPre}.}

\item{normIndex}{An integer equalling the index (position) of the
inside product whose mean valuation will be normalized to 1. Default
is 1, unless \sQuote{shares} sum to less than 1, in which case the default is
NA and an outside good is assumed to exist.}

\item{priceOutside}{price of the outside good. Equals 0 for logit and 1 for ces. Not used for aids.}

\item{priceStart}{For aids, a vector of length k who elements equal to an initial guess of the proportional change in price caused by the merger.
The default is to draw k random elements from a [0,1] uniform distribution. For ces and logit, the default is prices.}

\item{control.slopes}{A list of  \code{\link{optim}}  control parameters passed to the calibration routine optimizer (typically the \code{calcSlopes} method).}

\item{control.equ}{A list of  \code{\link[BB]{BBsolve}} control parameters passed to the non-linear equation solver (typically the \code{calcPrices} method).}

\item{labels}{A k-length vector of labels.}

\item{...}{Additional options to feed to the \code{\link[BB]{BBsolve}} optimizer used to solve for equilibrium prices.}
}
\value{
\code{bargaining_tariff} returns an instance of class \code{\linkS4class{TariffBargainingLogit}}
}
\description{
Simulate the effect of tariffs when firms play a Nash Bargaining game and consumer demand is Logit.
}
\details{
Let k denote the number of products produced by all firms.
Using price, and quantity, information for all products
in each market, as well as margin information for at least
one products in each market, \code{bargaining_tariff} is able to
recover the slopes and intercepts of a Logit  demand
system. These parameters are then used to simulate the price
effects of an \emph{ad valorem} tariff under the assumption that the firms are playing a
Nash Bargaining game.
}
\examples{
\donttest{
## Calibration and simulation results from a 10\% tariff on non-US beers "OTHER-LITE"
## and "OTHER-REG"
## Source: Epstein/Rubenfeld 2004, pg 80

prodNames <- c("BUD","OLD STYLE","MILLER","MILLER-LITE","OTHER-LITE","OTHER-REG")
owner <-c("BUD","OLD STYLE","MILLER","MILLER","OTHER-LITE","OTHER-REG")
price    <- c(.0441,.0328,.0409,.0396,.0387,.0497)
shares   <- c(.066,.172,.253,.187,.099,.223)
margins <- c(.3830,.5515,.5421,.5557,.4453,.3769) # margins in terms of price
tariff <- c(0,0,0,0,.1,.1)

names(price) <-
 names(shares) <-
 names(margins) <-
 prodNames


result.barg <- bargaining_tariff(demand = "logit",prices=price,shares=shares,
                                margins = margins,owner=owner,
                                 tariffPost = tariff, labels=prodNames)

print(result.barg)           # return predicted price change
summary(result.barg)         # summarize merger simulation
}
}
\references{
Simon P. Anderson, Andre de Palma, Brent Kreider, Tax incidence in differentiated product oligopoly,
Journal of Public Economics, Volume 81, Issue 2, 2001, Pages 173-192.
}
\seealso{
\code{\link{bertrand_tariff}} to simulate the effects of a tariff under a Bertrand pricing game and \code{\link{monopolistic_competition_tariff}} to simulate the effects of a tariff under monopolistic competition.
}
