% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extract_trends.R
\name{extract_trends}
\alias{extract_trends}
\title{Extract trends from time series objects}
\usage{
extract_trends(
  ts_data,
  methods = "stl",
  window = NULL,
  smoothing = NULL,
  band = NULL,
  align = NULL,
  params = list(),
  .quiet = FALSE
)
}
\arguments{
\item{ts_data}{A time series object (\code{ts}, \code{xts}, or \code{zoo}) or any object
convertible via tsbox.}

\item{methods}{Character vector of trend methods.
Options: \code{"hp"}, \code{"bk"}, \code{"cf"}, \code{"ma"}, \code{"stl"}, \code{"loess"}, \code{"spline"}, \code{"poly"},
\code{"bn"}, \code{"ucm"}, \code{"hamilton"}, \code{"spencer"}, \code{"ewma"}, \code{"wma"},
\code{"triangular"}, \code{"kernel"}, \code{"kalman"}, \code{"median"},
\code{"gaussian"}.
Default is \code{"stl"}.}

\item{window}{Unified window/period parameter for moving
average methods (ma, wma, triangular, stl, ewma, median, gaussian). Must be positive.
If NULL, uses frequency-appropriate defaults. For EWMA, specifies the window
size when using TTR's optimized implementation. Cannot be used simultaneously
with \code{smoothing} for EWMA method.}

\item{smoothing}{Unified smoothing parameter for smoothing
methods (hp, loess, spline, ewma, kernel, kalman).
For hp: use large values (1600+) or small values (0-1) that get converted.
For EWMA: specifies the alpha parameter (0-1) for traditional exponential smoothing.
Cannot be used simultaneously with \code{window} for EWMA method.
For kernel: multiplier of optimal bandwidth (1.0 = optimal, <1 = less smooth, >1 = more smooth).
For kalman: controls the ratio of measurement to process noise (higher = more smoothing).
For others: typically 0-1 range.}

\item{band}{Unified band parameter for bandpass filters
(bk, cf). Both values must be positive.
For bk/cf: Provide as \code{c(low, high)} where low/high are periods in quarters, e.g., \code{c(6, 32)}.}

\item{align}{Unified alignment parameter for moving average
methods (ma, wma, triangular, gaussian). Valid values: \code{"center"} (default, uses
surrounding values), \code{"right"} (causal, uses past values only), \code{"left"} (anti-causal,
uses future values only). Note: triangular only supports \code{"center"} and \code{"right"}.
If NULL, uses \code{"center"} as default.}

\item{params}{Optional list of method-specific parameters for fine control:
\itemize{
\item \strong{HP Filter}: \code{hp_onesided} (logical, default FALSE) - Use one-sided (real-time) filter instead of two-sided
\item \strong{Spline}: \code{spline_cv} (logical/NULL) - Cross-validation method: NULL (none), TRUE (leave-one-out), FALSE (GCV)
\item \strong{Polynomial}: \code{poly_degree} (integer, default 1), \code{poly_raw} (logical, default FALSE for orthogonal polynomials)
\item \strong{UCM}: \code{ucm_type} (character, default "level") - Model type: "level", "trend", or "BSM"
\item \strong{Others}: \code{bn_ar_order}, \code{hamilton_h}, \code{hamilton_p},
\code{kernel_type}, \code{kalman_measurement_noise}, \code{kalman_process_noise},
\code{median_endrule}, \code{gaussian_sigma}, \code{wma_weights}.
\item \strong{Note}: Alignment parameters (\code{ma_align}, \code{wma_align}, \code{triangular_align}, \code{gaussian_align})
can still be passed via \code{params} but it's recommended to use the unified \code{align} parameter instead.
}}

\item{.quiet}{If \code{TRUE}, suppress informational messages.}
}
\value{
If single method, returns a \code{ts} object. If multiple methods, returns
a named list of \code{ts} objects.
}
\description{
Extract trend components from time series objects using various econometric methods.
Designed for monthly and quarterly economic data analysis. Returns trend components
as time series objects or a list of time series.
}
\details{
This function focuses on monthly (frequency = 12) and quarterly (frequency = 4)
economic data. It uses established econometric methods with appropriate defaults:
\itemize{
\item \strong{HP Filter}: lambda=1600 (quarterly), lambda=14400 (monthly). Supports both two-sided and one-sided (real-time) variants
\item \strong{Baxter-King}: Bandpass filter for business cycles (6-32 quarters default)
\item \strong{Christiano-Fitzgerald}: Asymmetric bandpass filter
\item \strong{Moving Average}: Centered, frequency-appropriate windows
\item \strong{STL}: Seasonal-trend decomposition
\item \strong{Loess}: Local polynomial regression
\item \strong{Spline}: Smoothing splines
\item \strong{Polynomial}: Linear/polynomial trends
\item \strong{Beveridge-Nelson}: Permanent/transitory decomposition
\item \strong{UCM}: Unobserved Components Model (local level)
\item \strong{Hamilton}: Regression-based alternative to HP filter
\item \strong{Advanced MA}: EWMA with various implementations
\item \strong{Kernel Smoother}: Non-parametric regression with various kernel functions
\item \strong{Kalman Smoother}: Adaptive filtering for noisy time series
\item \strong{Median Filter}: Robust filtering using running medians to remove outliers
\item \strong{Gaussian Filter}: Weighted average with Gaussian (normal) density weights
}

\strong{Parameter Usage Notes}:
\itemize{
\item \strong{HP Filter}: Use \code{hp_onesided=TRUE} for real-time analysis or when future data should not
influence current estimates. One-sided filter is appropriate for nowcasting, policy analysis,
and avoiding look-ahead bias. Default two-sided filter is optimal for historical analysis.
\item \strong{EWMA}: Use either \code{window} (TTR optimization) OR \code{smoothing} (alpha parameter), not both
\item \strong{Kalman}: Use \code{smoothing} parameter or \code{params} list for fine control of noise parameters
\item \strong{Spline}: Use \code{spline_cv} to control cross-validation (NULL=none, TRUE=LOO-CV, FALSE=GCV)
\item \strong{Polynomial}: Use \code{poly_raw=FALSE} for orthogonal polynomials (more stable for degree > 2)
or \code{poly_raw=TRUE} for raw polynomials. Warning issued for degree > 3 (overfitting risk).
\item \strong{UCM}: Choose model type - "level" (simplest), "trend" (time-varying slope), or
"BSM" (with seasonal component, requires seasonal data)
}
}
\examples{
# Single method
hp_trend <- extract_trends(AirPassengers, methods = "hp")

# Multiple methods with unified smoothing
smooth_trends <- extract_trends(
  AirPassengers,
  methods = c("hp", "loess", "ewma"),
  smoothing = 0.3
)

# EWMA with window (uses TTR optimization)
ewma_window <- extract_trends(AirPassengers, methods = "ewma", window = 12)

# EWMA with alpha (traditional formula)
ewma_alpha <- extract_trends(AirPassengers, methods = "ewma", smoothing = 0.2)

# Moving averages with unified window
ma_trends <- extract_trends(
  AirPassengers,
  methods = c("ma", "wma", "triangular"),
  window = 8
)

# Bandpass filters with unified band
bp_trends <- extract_trends(
  AirPassengers,
  methods = c("bk", "cf"),
  band = c(6, 32)
)

# Moving average with right alignment (causal filter)
ma_causal <- extract_trends(
  AirPassengers,
  methods = "ma",
  window = 12,
  align = "right"
)

# Signal processing methods with specific parameters
finance_trends <- extract_trends(
  AirPassengers,
  methods = c("kalman", "gaussian"),
  window = 9,  # For Gaussian filter
  params = list(kalman_measurement_noise = 0.1)  # Kalman-specific parameter
)

# Spline with cross-validation options
spline_trends <- extract_trends(
  AirPassengers,
  methods = "spline",
  params = list(spline_cv = FALSE)  # Use GCV instead of default
)

# Polynomial with orthogonal vs raw polynomials
poly_trends <- extract_trends(
  AirPassengers,
  methods = "poly",
  params = list(poly_degree = 2, poly_raw = FALSE)  # Orthogonal (default)
)

# UCM with different model types
ucm_trends <- extract_trends(
  AirPassengers,
  methods = "ucm",
  params = list(ucm_type = "BSM")  # Basic Structural Model with seasonality
)

# HP Filter: One-sided (real-time) vs Two-sided (historical)
hp_realtime <- extract_trends(
  AirPassengers,
  methods = "hp",
  params = list(hp_onesided = TRUE)  # For nowcasting and real-time analysis
)

# Advanced: fine-tune specific methods
custom_trends <- extract_trends(
  AirPassengers,
  methods = c("median", "kalman"),
  window = 7,
  params = list(median_endrule = "constant")
)

}
