% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analysis_functions.R
\name{analyze_workloop}
\alias{analyze_workloop}
\title{Analyze work loop object to compute work and power output}
\usage{
analyze_workloop(x, simplify = FALSE, GR = 1, M = -1, vel_bf = 0.05, ...)
}
\arguments{
\item{x}{A \code{workloop} object of class \code{muscle_stim} that has been
passed through
\code{select_cycles}. See Details.}

\item{simplify}{Logical. If \code{FALSE}, the full analyzed workloop
object is returned. If \code{TRUE} a simpler table of net work and power
(by cycle) is returned.}

\item{GR}{Gear ratio, set to 1 by default}

\item{M}{Velocity multiplier, set adjust the sign of velocity. This parameter
should generally be either -1 (the default) or 1.}

\item{vel_bf}{Critical frequency (scalar) for low-pass filtering of velocity
via \code{signal::butter()}}

\item{...}{Additional arguments potentially passed down from
\code{read_analyze_wl()} or \code{read_analyze_wl_dir()}}
}
\value{
The function returns a \code{list} of class \code{analyzed_workloop}
that provides instantaneous velocity, a smoothed velocity, and computes work,
 instantaneous power, and net power from a work loop experiment. All data are
 organized by the cycle number and important metadata are stored as
 Attributes.

Within the \code{list}, each entry is labeled by cycle and includes:
\item{Time}{Time, in sec}
\item{Position}{Length change of the muscle, corrected for gear ratio, in mm}
\item{Force}{Force, corrected for gear ratio, in mN}
\item{Stim}{When stimulation occurs, on a binary scale}
\item{Cycle}{Cycle ID, as a letter}
\item{Inst_velocity}{Instantaneous velocity, computed from \code{Position}
change, reported in meters/sec}
\item{Filt_velocity}{Instantaneous velocity, after low-pass filtering, again
 in meter/sec}
\item{Inst_Power}{Instantaneous power, a product of \code{Force} and
\code{Filt_velocity}, reported in J}
\item{Percent_of_Cycle}{The percent of that particular cycle which has
elapsed}

In addition, the following information is stored in the
\code{analyzed_workloop} object's attributes:
\item{stimulus_frequency}{Frequency at which stimulus pulses occurred}
\item{cycle_frequency}{Frequency of oscillations (assuming sine wave
trajectory)}
\item{total_cycles}{Total number of oscillatory cycles (assuming sine wave
trajectory) that the muscle experienced.}
\item{cycle_def}{Specifies what part of the cycle is understood as the
beginning and end. There are currently three options:
'lo' for L0-to-L0;
'p2p' for peak-to-peak; and
't2t' for trough-to-trough}
\item{amplitude}{Amplitude of length change (assuming sine wave
trajectory)}
\item{phase}{Phase of the oscillatory cycle (in percent) at which stimulation
occurred. Somewhat experimental, please use with caution}
\item{position_inverted}{Logical; whether position inversion has been
applied)}
\item{units}{The units of measurement for each column in the object after
running this function. See Warning}
\item{sample_frequency}{Frequency at which samples were collected}
\item{header}{Additional information from the header}
\item{units_table}{Units from each Channel of the original ddf file}
\item{protocol_table}{Protocol in tabular format; taken from the original
ddf file}
\item{stim_table}{Specific info on stimulus protocol; taken from the original
ddf file}
\item{stimulus_pulses}{Number of sequential pulses within a stimulation
train}
\item{stimulus_offset}{Timing offset at which stimulus began}
\item{gear_ratio}{Gear ratio applied by this function}
\item{file_id}{File name}
\item{mtime}{Time at which file was last modified}
\item{retained_cycles}{Which cycles were retained, as numerics}
\item{summary}{Simple table showing work (in J) and net power (in W) for each
 cycle}
}
\description{
Compute work and power output from a work loop experiment on a per-cycle
basis.
}
\details{
Please note that \code{select_cycles()} must be run on data prior to
using this function. This function relies on the input \code{muscle_stim}
object being organized by cycle number.

The \code{muscle_stim} object (\code{x}) must be a \code{workloop},
preferably read in by one of our data import functions. Please see
documentation for \code{as_muscle_stim()} if you need to manually construct
a \code{muscle_stim} object from a non .ddf source.

The gear ratio (GR) and velocity multiplier (M) parameters can help correct
for issues related to the magnitude and sign of data collection. By default,
they are set to apply no gear ratio adjustment and to positivize velocity.
Instantaneous velocity is often noisy and the \code{vel_bf} parameter allows
for low-pass filtering of velocity data. See \code{signal::butter()} and
\code{signal::filtfilt()} for details of how filtering is achieved.

Please also be careful with units! Se Warning section below.
}
\section{Warning}{

Most systems we have encountered record Position data in millimeters
and Force in millinewtons, and therefore this function assumes data are
recorded in those units. Through a series of internal conversions, this
function computes velocity in meters/sec, work in Joules, and power in
Watts. If your raw data do not originate in millimeters and millinewtons,
please transform your data accordingly and ignore what you see in the
attribute \code{units}.
}

\examples{

library(workloopR)

# import the workloop.ddf file included in workloopR
wl_dat <-read_ddf(system.file("extdata", "workloop.ddf",
                              package = 'workloopR'),
                  phase_from_peak = TRUE)

# select cycles 3 through 5 via the peak-to-peak definition
wl_selected <- select_cycles(wl_dat, cycle_def = "p2p", keep_cycles = 3:5)

# run the analysis function and get the full object
wl_analyzed <- analyze_workloop(wl_selected, GR = 2)

# print methods give a short summary
print(wl_analyzed)

# summary provides a bit more detail
summary(wl_analyzed)

# run the analysis but get the simplified version
wl_analyzed_simple <- analyze_workloop(wl_selected, simplify = TRUE, GR = 2)

}
\references{
Josephson RK. 1985. Mechanical Power output from Striated Muscle
 during Cyclic Contraction. Journal of Experimental Biology 114: 493-512.
}
\seealso{
\code{\link{read_ddf}},
\code{\link{read_analyze_wl}},
\code{\link{select_cycles}}

Other data analyses: 
\code{\link{isometric_timing}()},
\code{\link{read_analyze_wl_dir}()},
\code{\link{read_analyze_wl}()}

Other workloop functions: 
\code{\link{fix_GR}()},
\code{\link{get_wl_metadata}()},
\code{\link{invert_position}()},
\code{\link{read_analyze_wl_dir}()},
\code{\link{read_analyze_wl}()},
\code{\link{select_cycles}()},
\code{\link{summarize_wl_trials}()},
\code{\link{time_correct}()}
}
\author{
Vikram B. Baliga and Shreeram Senthivasan
}
\concept{data analyses}
\concept{workloop functions}
